package ru.yandex.stockpile.api.grpc.handler;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Throwables;
import io.grpc.MethodDescriptor;
import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.slog.LogDataIterator;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.ResolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaRecord;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.StockpileServiceGrpc;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteLogResponse;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.server.shard.MetricArchives;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.stockpile.api.grpc.handler.Handlers.classifyError;
import static ru.yandex.stockpile.api.grpc.handler.Handlers.logError;

/**
 * @author Vladimir Gordiychuk
 */
public class WriteLogHandler extends ShardRequestHandler<TWriteLogRequest, TWriteLogResponse> {
    private static final Logger logger = LoggerFactory.getLogger(WriteLogHandler.class);

    private static final MethodDescriptor<TWriteLogRequest, TWriteLogResponse> ENDPOINT =
        StockpileServiceGrpc.getWriteLogMethod();

    public WriteLogHandler(StockpileLocalShards shards) {
        super(shards);
    }

    @Override
    public CompletableFuture<TWriteLogResponse> unaryCall(StockpileShard shard, TWriteLogRequest request) {
        var state = shard.getLoadState();
        if (state != StockpileShard.LoadState.DONE) {
            return completedFuture(response(EStockpileStatusCode.SHARD_NOT_READY, state.name()));
        }

        return write(shard, parse(request));
    }

    @Override
    protected int shardId(TWriteLogRequest request) {
        return request.getShardId();
    }

    @Override
    protected TWriteLogResponse response(EStockpileStatusCode status, String details) {
        return TWriteLogResponse.newBuilder()
                .setStatus(status)
                .setStatusMessage(details)
                .build();
    }

    private CompletableFuture<TWriteLogResponse> write(StockpileShard shard, List<Log> logs) {
        var futures = new ArrayList<CompletableFuture<?>>();
        long metrics = 0;
        for (var log : logs) {
            metrics += log.metrics;
            var id = log.requestId;
            logger.debug("Receive log {} for shard {}", id, shard.shardId);
            var future = shard.pushBatch(new StockpileWriteRequest(log.archiveByLocalId, null, id.producerId, id.producerSeqNo));
            futures.add(future);
        }
        long finalMetrics = metrics;
        return CompletableFutures.allOfVoid(futures)
            .handle((ignore, e) -> {
                if (e == null) {
                    return TWriteLogResponse.newBuilder()
                            .setStatus(EStockpileStatusCode.OK)
                            .build();
                }

                logError(this, DataSize.shortString(finalMetrics) + " metrics", e);
                EStockpileStatusCode code = classifyError(e);

                return TWriteLogResponse.newBuilder()
                        .setStatus(code)
                        .setStatusMessage(Throwables.getStackTraceAsString(e))
                        .build();
            });
    }

    public List<Log> parse(TWriteLogRequest request) {
        try {
            int shardId = request.getShardId();
            var index = LogsIndexSerializer.deserialize(ByteStrings.toByteBuf(request.getIndex()));
            var result = new ArrayList<Log>();
            var idxByRequestId = new Object2IntOpenHashMap<RequestId>(index.getSize());
            idxByRequestId.defaultReturnValue(-1);
            ByteBuf content = ByteStrings.toByteBuf(request.getContent());
            try {
                for (int idx = 0; idx < index.getSize(); idx++) {
                    var meta = content.readSlice(index.getMetaSize(idx));
                    var data = content.readSlice(index.getDataSize(idx));
                    var log = parse(shardId, meta, data);
                    var prevIdx = idxByRequestId.getInt(log.requestId);
                    if (prevIdx == -1) {
                        idxByRequestId.put(log.requestId, result.size());
                        result.add(log);
                    } else {
                        result.get(prevIdx).combine(log);
                    }
                }
                return result;
            } finally {
                content.release();
            }
        } catch (RuntimeException e) {
            throw new StockpileRuntimeException(EStockpileStatusCode.INVALID_REQUEST, e);
        }
    }

    public Log parse(int shardId, ByteBuf meta, ByteBuf data) {
        var header = new ResolvedLogMetaHeader(meta);
        var metaRecord = new ResolvedLogMetaRecord();
        var result = new Log(shardId, header);
        try (var metaIt = new ResolvedLogMetaIteratorImpl(header, meta);
             var dataIt = LogDataIterator.create(data))
        {
            while (metaIt.next(metaRecord)) {
                var archive = result.archiveRef(metaRecord.localId);
                archive.setDecimPolicyId(header.decimPolicy.getNumber());
                archive.ensureCapacity(metaRecord.dataSize);
                // TODO: accept project from slog header(@gordiychuk)
                archive.setOwnerProjectIdEnum(EProjectId.GOLOVAN);
                dataIt.next(archive);
            }
        }
        return result;
    }

    @Override
    public MethodDescriptor<TWriteLogRequest, TWriteLogResponse> descriptor() {
        return ENDPOINT;
    }

    @Override
    public EStockpileStatusCode getStatusCode(TWriteLogResponse response) {
        return response.getStatus();
    }

    private static class Log {
        private final int shardId;
        private final RequestId requestId;
        public int metrics;
        public int records;
        public final Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId;

        public Log(int shardId, ResolvedLogMetaHeader header) {
            this.shardId = shardId;
            this.requestId = new RequestId(header.producerId, header.producerSeqNo);
            this.metrics = header.metricsCount;
            this.records = header.pointsCount;
            this.archiveByLocalId = new Long2ObjectOpenHashMap<>(header.metricsCount);
        }

        public MetricArchiveMutable archiveRef(long localId) {
            var archive = archiveByLocalId.get(localId);
            if (archive == null) {
                archive = new MetricArchiveMutable();
                archiveByLocalId.put(localId, archive);
            }
            return archive;
        }

        public void combine(Log other) {
            this.metrics += other.metrics;
            this.records += other.records;
            var it = other.archiveByLocalId.long2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var entry = it.next();
                long localId = entry.getLongKey();
                var prev = archiveByLocalId.get(localId);
                if (prev == null) {
                    archiveByLocalId.put(localId, entry.getValue());
                } else {
                    MetricArchives.typeSafeAppend(shardId, localId, prev, entry.getValue());
                }
            }
        }
    }

    private record RequestId(int producerId, long producerSeqNo) {
    }
}
