package ru.yandex.stockpile.api.grpc.mem;

import java.io.IOException;
import java.io.InputStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.WireFormat;
import io.grpc.MethodDescriptor;

import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.api.grpc.TCommandRequestParser;
import ru.yandex.stockpile.server.shard.StockpileWriteRequest;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ShardBatchRequestMarshaller implements MethodDescriptor.Marshaller<ShardBatchRequest> {

    @Override
    public InputStream stream(ShardBatchRequest value) {
        throw new UnsupportedOperationException("Can't be use on server side");
    }

    @Override
    public ShardBatchRequest parse(InputStream stream) {
        long startNanos = System.nanoTime();
        var input = CodedInputStream.newInstance(stream);
        var point = RecyclableAggrPoint.newInstance();
        try {
            boolean done = false;
            int shardId = 0;
            long recordCount = 0;
            var builder = StockpileWriteRequest.newBuilder();
            while (!done) {
                int tag = input.readTag();
                if (tag == 0) {
                    break;
                }

                switch (WireFormat.getTagFieldNumber(tag)) {
                    case TShardCommandRequest.SHARDID_FIELD_NUMBER:
                        shardId = input.readInt32();
                        break;
                    case TShardCommandRequest.COMMANDS_FIELD_NUMBER:
                        int length = input.readRawVarint32();
                        final int oldLimit = input.pushLimit(length);

                        recordCount += TCommandRequestParser.parse(builder, point, input);

                        input.checkLastTagWas(0);
                        input.popLimit(oldLimit);
                        break;
                    default:
                        if (!input.skipField(tag)) {
                            done = true;
                        }
                        break;
                }
            }

            return new ShardBatchRequest(shardId, builder.build(), recordCount, System.nanoTime() - startNanos);
        } catch (IOException e) {
            throw new StockpileRuntimeException(EStockpileStatusCode.CORRUPTED_BINARY_DATA, e.getMessage(), e);
        } finally {
            point.recycle();
        }
    }
}
