package ru.yandex.stockpile.api.read;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.TextFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.operation.OperationsPipeline;
import ru.yandex.solomon.math.operation.OperationsPipelineMap;
import ru.yandex.solomon.math.operation.OperationsPipelineReduced;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.shard.StockpileMetricReadRequest;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
@Component
@Import({
        StockpileReadApiMetrics.class
})
public class StockpileReadApi {
    private static final Logger logger = LoggerFactory.getLogger(StockpileReadApi.class);
    private static final long READ_AGE_THRESHOLD = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(360 * 2);
    // TODO: extract to config after combine all reads into this class (gordiychuk@)
    private static final int MAX_READ_INFLIGHTS = 15_000;

    private final StockpileReadApiMetrics metrics;
    private final StockpileReadApiMetrics.ProjectReadMetrics readMetrics;

    public StockpileReadApi(StockpileReadApiMetrics metrics) {
        this.metrics = metrics;
        this.readMetrics = metrics.getProjectMetrics(EProjectId.UNKNOWN.name());
    }

    public OperationsPipeline<MetricId> readMany(StockpileShard shard, TReadManyRequest request) {
        Interval interval = parseInterval(request);
        long deadline = parseDeadline(request);

        ensureDeadlineNotExpired(deadline);
        if (interval.getBeginMillis() <= READ_AGE_THRESHOLD) {
            logger.info("Big range read {}, request {}", interval, TextFormat.shortDebugString(request));
        }

        readMetrics.updateReadOffset(interval, request.getLocalIdsCount());
        if (metrics.getReadMetricsInFlight() + request.getLocalIdsCount() > MAX_READ_INFLIGHTS) {
            return new OperationsPipelineReduced<>(
                    CompletableFuture.failedFuture(
                            new StockpileRuntimeException(EStockpileStatusCode.RESOURCE_EXHAUSTED,
                                    "too many inFlight reads")));
        }

        List<StockpileMetricReadRequest> readTasks = new ArrayList<>(request.getLocalIdsCount());
        List<CompletableFuture<Metric<MetricId>>> results = new ArrayList<>(request.getLocalIdsCount());
        for (long localId : request.getLocalIdsList()) {
            if (!StockpileLocalId.isValid(localId)) {
                return new OperationsPipelineReduced<>(
                        CompletableFuture.failedFuture(
                                new StockpileRuntimeException(
                                        EStockpileStatusCode.INVALID_REQUEST, "localId not valid: " + localId)));
            }

            MetricId metricId = MetricId.newBuilder()
                    .setShardId(request.getShardId())
                    .setLocalId(localId)
                    .build();

            StockpileMetricReadRequest task = StockpileMetricReadRequest.newBuilder()
                    .setLocalId(localId)
                    .setShardId(metricId.getShardId())
                    .setFromMillis(interval.getBeginMillis())
                    // current shard implementation filter time as from inclusive to inclusive, but gRPC as from inclusive to exclusive
                    .setToMillis(interval.getEndMillis() - 1)
                    .setDeadline(deadline)
                    .setProducer(request.getProducer())
                    .build();

            readTasks.add(task);
            results.add(task.getFuture()
                    .thenApply(result -> {
                        ensureDeadlineNotExpired(deadline);
                        return new Metric<>(metricId, result.getHeader().getType(), result.getHeader().getOwnerProject(), result.getTimeseries());
                    }));
        }
        addShardReads(shard, readTasks);
        return new OperationsPipelineMap<>(results).apply(interval, request.getOperationsList());
    }

    protected boolean isDeadlineExceeded(long deadline) {
        return deadline != 0 && System.currentTimeMillis() + 10 >= deadline;
    }

    private void ensureDeadlineNotExpired(long deadline) {
        if (isDeadlineExceeded(deadline)) {
            throw new StockpileRuntimeException(EStockpileStatusCode.DEADLINE_EXCEEDED, false);
        }
    }

    private void addShardReads(StockpileShard shard, List<StockpileMetricReadRequest> requests) {
        readMetrics.started(requests.size());
        long startNanos = System.nanoTime();
        for (StockpileMetricReadRequest request : requests) {
            request.getFuture().whenComplete((response, e) -> {
                if (e != null) {
                    readMetrics.failed(1);
                } else {
                    long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                    readMetrics.completed(elapsedMillis, response.getTimeseries().getRecordCount());
                }
            });
        }

        shard.read(requests);
    }

    private Interval parseInterval(TReadManyRequest request) {
        long fromMillis = request.getFromMillis();
        long toMillis = request.getToMillis() == 0
                ? System.currentTimeMillis()
                : request.getToMillis();
        return Interval.millis(fromMillis, toMillis);
    }

    private long parseDeadline(TReadManyRequest request) {
        return request.getDeadline() == 0
                ? System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(30)
                : request.getDeadline();
    }
}
