package ru.yandex.stockpile.api.read;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@Component
public class StockpileReadApiMetrics implements MetricSupplier {
    private final ConcurrentHashMap<String, ProjectReadMetrics> metrics = new ConcurrentHashMap<>();

    public ProjectReadMetrics getProjectMetrics(String projectId) {
        return metrics.computeIfAbsent(projectId, ProjectReadMetrics::new);
    }

    public long getReadMetricsInFlight() {
        long result = 0;
        for (ProjectReadMetrics read : metrics.values()) {
            result += read.getReadInFlight();
        }
        return result;
    }

    @Override
    public int estimateCount() {
        return metrics.values()
                .stream()
                .mapToInt(ProjectReadMetrics::estimateCount)
                .sum();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        ProjectReadMetrics total = new ProjectReadMetrics("total");
        for (ProjectReadMetrics projectMetrics : metrics.values()) {
            projectMetrics.append(tsMillis, commonLabels, consumer);
            total.combine(projectMetrics);
        }
        total.append(tsMillis, commonLabels, consumer);
    }

    public static class ProjectReadMetrics implements MetricSupplier {
        private final String projectId;
        private final MetricRegistry registry;
        private final Histogram metricOffsetHours;
        private final Histogram metricElapsedTime;
        private final Histogram recordElapsedTime;
        private final Rate metricStarted;
        private final Rate metricCompleted;
        private final Rate metricFailed;
        private final LazyGaugeInt64 metricInFlight;

        public ProjectReadMetrics(String projectId) {
            this.projectId = projectId;
            this.registry = new MetricRegistry();
            this.metricOffsetHours = this.registry.histogramRate("stockpile.read.sensors.offsetHours",
                Histograms.exponential(18, 2, 1));
            this.metricElapsedTime = this.registry.histogramRate("stockpile.read.sensors.elapsedTimeMillis",
                    Histograms.exponential(18, 2, 1));
            this.recordElapsedTime = this.registry.histogramRate("stockpile.read.records.elapsedTimeMillis",
                    Histograms.exponential(18, 2, 1));

            this.metricStarted = this.registry.rate("stockpile.read.sensors.started");
            this.metricCompleted = this.registry.rate("stockpile.read.sensors.completed");
            this.metricFailed = this.registry.rate("stockpile.read.sensors.failed");
            this.metricInFlight = this.registry.lazyGaugeInt64("stockpile.read.sensors.inFlight",
                    () -> metricStarted.get() - metricCompleted.get() - metricFailed.get());
        }

        public void updateReadOffset(Interval interval, int count) {
            long now = System.currentTimeMillis();
            updateReadOffset(now - interval.getBeginMillis(), count);
        }

        public void updateReadOffset(long offsetMillis, int count) {
            metricOffsetHours.record(TimeUnit.MILLISECONDS.toHours(offsetMillis), count);
        }

        public long getReadInFlight() {
            return metricInFlight.get();
        }

        public void started(int count) {
            metricStarted.add(count);
        }

        public void completed(long elapsedTimeMillis, long records) {
            metricCompleted.inc();
            metricElapsedTime.record(elapsedTimeMillis);
            recordElapsedTime.record(elapsedTimeMillis, records);
        }

        public void failed(int count) {
            metricFailed.add(count);
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            commonLabels = commonLabels.add("projectId", projectId);
            registry.append(tsMillis, commonLabels, consumer);
        }

        public void combine(ProjectReadMetrics target) {
            metricOffsetHours.combine(target.metricOffsetHours);
            metricElapsedTime.combine(target.metricElapsedTime);
            recordElapsedTime.combine(target.recordElapsedTime);
            metricStarted.combine(target.metricStarted);
            metricCompleted.combine(target.metricCompleted);
            metricFailed.combine(target.metricFailed);
        }
    }
}
