package ru.yandex.stockpile.cluster.balancer;

import java.util.concurrent.CompletableFuture;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KvTabletIdAndGen;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class KvTableClientImpl implements KvTabletClient {
    private final KikimrKvClient kvClient;
    private final KvTabletsMapping mapping;

    public KvTableClientImpl(KikimrKvClient kvClient, KvTabletsMapping mapping) {
        this.kvClient = kvClient;
        this.mapping = mapping;
    }

    @Override
    public int getShardCount() {
        return mapping.getShardCount();
    }

    @Override
    public CompletableFuture<KvTabletIdAndGen> incrementGeneration(int shardId) {
        return mapping.getReadyFuture()
            .thenCompose(ignore -> resolveTabletId(shardId))
            .thenCompose(this::incTabletGeneration);
    }

    private CompletableFuture<Long> resolveTabletId(int shardId) {
        long tabletId = mapping.getTabletId(shardId);
        if (tabletId != 0) {
            return completedFuture(tabletId);
        }

        return mapping.forceReload().thenApply(ignore -> mapping.getTabletId(shardId));
    }

    private CompletableFuture<KvTabletIdAndGen> incTabletGeneration(long tabletId) {
        if (tabletId == KvTabletsMapping.UNKNOWN_TABLET_ID) {
            return CompletableFuture.failedFuture(new IllegalStateException("Unable to inc generation for tableId: " + tabletId));
        }

        return kvClient.incrementGeneration(tabletId, 0)
                .thenApply(gen -> new KvTabletIdAndGen(tabletId, gen));
    }
}
