package ru.yandex.stockpile.cluster.balancer;

import java.nio.file.Path;
import java.time.Clock;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.annotation.Order;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.balancer.TotalShardCounter;
import ru.yandex.solomon.balancer.TotalShardCounterImpl;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.balancer.remote.RemoteClusterImpl;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.www.BalancerController;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.util.file.SimpleFileStorage;
import ru.yandex.stockpile.cluster.StockpileLockServiceContext;
import ru.yandex.stockpile.cluster.balancer.client.GrpcStockpileBalancerClient;
import ru.yandex.stockpile.cluster.balancer.client.StockpileBalancerClient;
import ru.yandex.stockpile.cluster.balancer.dao.KvStockpileBalancerDao;
import ru.yandex.stockpile.cluster.balancer.leader.StockpileRemoteNodeClient;
import ru.yandex.stockpile.cluster.balancer.leader.StockpileResource;
import ru.yandex.stockpile.cluster.balancer.server.StockpileBalancerServiceGrpc;
import ru.yandex.stockpile.cluster.discovery.StockpileDiscoveryContext;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;
import ru.yandex.stockpile.server.shard.StockpileBalancerExecutor;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileScheduledExecutor;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileShardGlobals;

/**
 * @author Vladimir Gordiychuk
 */
@Import({
    StockpileLockServiceContext.class,
    StockpileDiscoveryContext.class,
    ManagerWriterContext.class
})
@Configuration
public class StockpileBalancerContext {
    private static final Path CACHE_DIR = Path.of("cache");

    private final Clock clock;
    private final DistributedLock lock;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;

    public StockpileBalancerContext(
        Clock clock,
        @StockpileBalancerLock DistributedLock lock,
        @StockpileScheduledExecutor ScheduledExecutorService timer,
        @StockpileBalancerExecutor ExecutorService executor)
    {
        this.clock = clock;
        this.lock = lock;
        this.timer = timer;
        this.executor = executor;
    }

    @Bean
    public StockpileBalancerClient balancerClient(ClusterDiscovery<GrpcTransport> discovery) {
        return new GrpcStockpileBalancerClient(discovery);
    }

    @Bean
    public KvTabletClient kvTabletClient(KikimrKvClient kvClient, KvTabletsMapping mapping) {
        return new KvTableClientImpl(kvClient, mapping);
    }

    @Bean
    public KvStockpileBalancerDao kvStockpileBalancerDao(KikimrKvClient kvClient, TKikimrClientConfig config) {
        return new KvStockpileBalancerDao(kvClient, config.getSchemaRoot() + "/System");
    }

    @Bean
    public StockpileRemoteNodeClient stockpileRemoteNodeClient(StockpileBalancerClient client, KvTabletClient kvClient) {
        return new StockpileRemoteNodeClient(client, kvClient);
    }

    @Bean
    public RemoteCluster remoteCluster(RemoteNodeClient client) {
        return new RemoteClusterImpl(client, timer, clock);
    }

    @Bean
    public ShardsHolder shardsHolder(KvTabletClient tabletClient) {
        return new StockpileShardsHolder(tabletClient);
    }

    @Bean
    public TotalShardCounter totalShardCounter() {
        return new TotalShardCounterImpl();
    }

    @Bean
    public BalancerHolderImpl stockpileBalancer(
        RemoteCluster cluster,
        ShardsHolder shardsHolder,
        TotalShardCounter shardCounter,
        KvStockpileBalancerDao dao)
    {
        return new BalancerHolderImpl(
            clock,
            lock,
            StockpileResource.RESOURCES,
            cluster,
            executor,
            timer,
            shardsHolder,
            shardCounter,
            dao);
    }

    @Bean
    public StockpileLocalShardsState localShardsState(StockpileLocalShards shards, StockpileShardGlobals globals) {
        var storage = new SimpleFileStorage(CACHE_DIR);
        return new StockpileLocalShardsState(storage, shards, globals, lock, clock);
    }

    @Bean
    public StockpileBalancerServiceGrpc balancerServiceGrpc(StockpileLocalShardsState localShards) {
        return new StockpileBalancerServiceGrpc(localShards);
    }

    @Bean
    public StockpileBalancerServiceGrpc.Proxy balancerServiceGrpcProxy(StockpileBalancerServiceGrpc service) {
        return new StockpileBalancerServiceGrpc.Proxy(service);
    }

    @Bean
    public Bootstrap balancerBootstrap(BalancerHolderImpl balancer, StockpileLocalShardsState localState) {
        return new Bootstrap(balancer, localState);
    }

    @Bean
    public BalancerController balancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            BalancerHolderImpl balancer,
            HttpServerConfig config)
    {
        return new BalancerController(
                authenticator,
                authorizer,
                balancer,
                StockpileShard.class,
                config.getPort());
    }

    @Bean
    public RootLink balancerLink() {
        return new RootLink("/balancer", "Balancer");
    }

    public static class Bootstrap {
        private final BalancerHolderImpl balancer;
        private final StockpileLocalShardsState localState;

        public Bootstrap(
                BalancerHolderImpl balancer,
                StockpileLocalShardsState localState)
        {
            this.balancer = balancer;
            this.localState = localState;
        }

        @PostConstruct
        public void startupHandler() {
            balancer.start();
        }

        @Order(value = 0)
        @PreDestroy
        public void shutdownHandler() {
            balancer.close();
            localState.close();
        }

    }
}
