package ru.yandex.stockpile.cluster.balancer;

import java.time.Clock;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.Tasks;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.util.file.FileStorage;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.stockpile.internal.api.TAssignShardRequest;
import ru.yandex.stockpile.internal.api.TAssignShardResponse;
import ru.yandex.stockpile.internal.api.TPingRequest;
import ru.yandex.stockpile.internal.api.TPingResponse;
import ru.yandex.stockpile.internal.api.TShardAssignment;
import ru.yandex.stockpile.internal.api.TUnassignShardRequest;
import ru.yandex.stockpile.internal.api.TUnassignShardResponse;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;
import ru.yandex.stockpile.server.shard.StockpileShard.LoadState;
import ru.yandex.stockpile.server.shard.StockpileShardGlobals;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileLocalShardsState implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(StockpileLocalShardsState.class);
    public static final String ASSIGNMENTS_STATE_FILE = "stockpile.assignments.state";

    private final StockpileLocalShards shards;
    private final StockpileShardGlobals globals;
    private final DistributedLock leader;
    private final StockpileNodeSummary currentNode;
    private final FileStorage localStorage;
    private final ObjectMapper mapper = new ObjectMapper();
    private final Tasks flushStateTasks = new Tasks();
    private volatile boolean closed;

    // save shard assignments
    public StockpileLocalShardsState(
        FileStorage localStorage,
        StockpileLocalShards shards,
        StockpileShardGlobals globals,
        DistributedLock leader,
        Clock clock)
    {
        this.shards = shards;
        this.globals = globals;
        this.leader = leader;
        this.localStorage = localStorage;
        this.currentNode = new StockpileNodeSummary(clock, shards);
        this.restoreState();
    }

    public CompletableFuture<TPingResponse> ping(TPingRequest request) {
        return ensureLeaderOwnership(request.getLeaderSeqNo())
                .thenApply(node -> {
                    ensureNotClosed();
                    ensureDeadlineNotExpired(request.getExpiredAt());
                    logger.debug("Receive ping from {}", node);

                    shards.ensureCapacity(request.getShardCount());
                    return TPingResponse.newBuilder()
                            .setNode(HostUtils.getFqdn())
                            .setNodeSummary(currentNode.prepareNodeSummary())
                            .addAllShardSummary(currentNode.prepareShardsSummary())
                            .build();
                });
    }

    public CompletableFuture<TAssignShardResponse> assignShard(TAssignShardRequest request) {
        return ensureLeaderOwnership(request.getLeaderSeqNo())
            .thenApply(node -> {
                ensureNotClosed();
                ensureDeadlineNotExpired(request.getExpiredAt());
                TShardAssignment assignment = request.getAssignment();
                String debug = TextFormat.shortDebugString(assignment) + " from " + node;
                logger.info("Receive shard assignment {}", debug);

                StockpileShard prev = shards.getShardById(assignment.getShardId());
                if (prev != null) {
                    prev.stop();
                    shards.remove(prev);
                }

                StockpileShard shard = new StockpileShard(
                    globals,
                    assignment.getShardId(),
                    assignment.getTabletId(),
                    assignment.getTabletGeneration(),
                    debug);

                if (!shards.addShard(shard)) {
                    throw Status.FAILED_PRECONDITION
                        .withDescription("Shard " + assignment.getShardId() + " already assigned by another thread")
                        .asRuntimeException();
                }

                shard.start();
                return TAssignShardResponse.getDefaultInstance();
            }).whenComplete((ignore, e) -> globals.executorService.submit(this::flushState));
    }

    public CompletableFuture<TUnassignShardResponse> unassignShard(TUnassignShardRequest request) {
        return ensureLeaderOwnership(request.getLeaderSeqNo())
            .thenCompose(node -> {
                ensureNotClosed();
                ensureDeadlineNotExpired(request.getExpiredAt());
                String debug = TextFormat.shortDebugString(request) + " from " + node;
                logger.info("Receive shard unassign {}", debug);

                StockpileShard shard = shards.getShardById(request.getShardId());
                if (shard == null) {
                    return completedFuture(TUnassignShardResponse.getDefaultInstance());
                }

                if (!request.getGraceful() || shard.getLoadState() != LoadState.DONE) {
                    shard.stop();
                    shards.remove(shard);
                    return completedFuture(TUnassignShardResponse.getDefaultInstance());
                }

                return shard.forceSnapshot()
                    .handle((ignore, e) -> {
                        if (e != null) {
                            logger.warn("Failed graceful unassign {}", debug, e);
                        }

                        shard.stop();
                        shards.remove(shard);
                        return TUnassignShardResponse.getDefaultInstance();
                    });
            }).whenComplete((ignore, e) -> globals.executorService.submit(this::flushState));
    }

    private CompletableFuture<String> ensureLeaderOwnership(long seqNo) {
        return leader.getLockDetail(seqNo)
            .thenApply(detail -> {
                if (detail.isEmpty()) {
                    throw Status.ABORTED
                        .withDescription("Reject because leader ownership expired")
                        .asRuntimeException();
                }

                if (Long.compareUnsigned(seqNo, detail.get().seqNo()) != 0) {
                    throw Status.ABORTED
                        .withDescription("Rejected, seqNo mismatch("
                            + seqNo
                            + " != "
                            + detail.get().seqNo()
                            + "), leader now "
                            + detail.get().owner())
                        .asRuntimeException();
                }

                return detail.get().owner();
            });
    }

    private void ensureDeadlineNotExpired(long expiredAt) {
        if (expiredAt == 0) {
            return;
        }

        if (System.currentTimeMillis() + 200L >= expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    private void ensureNotClosed() {
        if (closed) {
            throw Status.UNAVAILABLE.withDescription("shutting down").asRuntimeException();
        }
    }

    private void flushState() {
        if (!flushStateTasks.addTask()) {
            return;
        }

        while (flushStateTasks.fetchTask()) {
            saveStateToFile(shards.stream()
                    .map(Assignment::of)
                    .collect(Collectors.toList()));
        }
    }

    private void restoreState() {
        try {
            List<Assignment> assignments = loadStateFromFile();
            for (var assignment : assignments) {
                StockpileShard shard = new StockpileShard(
                        globals,
                        assignment.shardId,
                        assignment.tabletId,
                        assignment.tabletGeneration,
                        "restored from " + assignment);

                if (shards.addShard(shard)) {
                    logger.info("Restore assignment {}", assignment);
                    shard.start();
                }
            }
        } catch (Throwable e) {
            logger.error("Restore assignments state failed", e);
        }
    }

    private void saveStateToFile(List<Assignment> config) {
        try {
            localStorage.save(ASSIGNMENTS_STATE_FILE, config, mapper::writeValueAsString);
        } catch (Throwable e) {
            logger.error("Error while writing state file " + ASSIGNMENTS_STATE_FILE, e);
        }
    }

    private List<Assignment> loadStateFromFile() {
        try {
            List<Assignment> assignments = localStorage.load(
                ASSIGNMENTS_STATE_FILE,
                state -> mapper.readValue(state, new TypeReference<List<Assignment>>() {}));

            return Objects.requireNonNullElseGet(assignments, List::of);
        } catch (Throwable e) {
            logger.error("Error while reading state file " + localStorage + "/" + ASSIGNMENTS_STATE_FILE, e);
            return List.of();
        }
    }

    @Override
    public void close() {
        closed = true;
    }

    private static class Assignment {
        public int shardId;
        public long tabletId;
        public long tabletGeneration;

        public static Assignment of(StockpileShard shard) {
            Assignment result = new Assignment();
            result.shardId = shard.shardId;
            result.tabletId = shard.kvTabletId;
            result.tabletGeneration = shard.getGeneration();
            return result;
        }

        @Override
        public String toString() {
            return "Assignment{" +
                    "shardId=" + shardId +
                    ", tabletId=" + tabletId +
                    ", tabletGeneration=" + tabletGeneration +
                    '}';
        }
    }

}
