package ru.yandex.stockpile.cluster.balancer;

import java.time.Clock;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.stockpile.internal.api.TNodeSummary;
import ru.yandex.stockpile.internal.api.TShardSummary;
import ru.yandex.stockpile.server.shard.StockpileLocalShards;
import ru.yandex.stockpile.server.shard.StockpileShard;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileNodeSummary {
    private static final Logger logger = LoggerFactory.getLogger(StockpileNodeSummary.class);
    private final Clock clock;
    private final StockpileLocalShards shards;
    private final long startedAt;

    public StockpileNodeSummary(Clock clock, StockpileLocalShards shards) {
        this.startedAt = clock.millis();
        this.clock = clock;
        this.shards = shards;
    }

    public TNodeSummary prepareNodeSummary() {
        try {
            return TNodeSummary.newBuilder()
                    .setMemoryBytes(ProcSelfMon.getRssBytes())
                    .setUtimeMillis(ProcSelfMon.getUtimeMs())
                    .setUpTimeMillis(clock.millis() - startedAt)
                    .build();
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
            return TNodeSummary.getDefaultInstance();
        }
    }

    public List<TShardSummary> prepareShardsSummary() {
        return shards.stream()
                .map(this::prepareShardSummary)
                .collect(Collectors.toList());
    }

    private TShardSummary prepareShardSummary(StockpileShard shard) {
        TShardSummary.Builder summary = TShardSummary.newBuilder();
        summary.setShardId(shard.shardId);
        switch (shard.getLoadState()) {
            case INIT:
                summary.setStatus(TShardSummary.EStatus.INIT);
                break;
            case DONE:
                summary.setStatus(TShardSummary.EStatus.READY);
                break;
            default:
                summary.setStatus(TShardSummary.EStatus.LOADING);
        }

        summary.setUpTimeMillis(clock.millis() - shard.stockpileShardCreatedInstantMillis);
        summary.setUtimeMillis(TimeUnit.NANOSECONDS.toMillis(Math.round(shard.metrics().utimeNanos.getRate(TimeUnit.SECONDS))));
        summary.setMemoryBytes(shard.memorySizeIncludingSelf());
        summary.setMetricsReadRate(shard.metrics.read.avgMetricsRps.getRate(TimeUnit.SECONDS));
        summary.setRecordsWriteRate(shard.metrics.write.avgRecordsRps.getRate(TimeUnit.SECONDS));
        return summary.build();
    }

}
