package ru.yandex.stockpile.cluster.balancer.client;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.stockpile.internal.api.StockpileBalancerServiceGrpc;
import ru.yandex.stockpile.internal.api.TAssignShardRequest;
import ru.yandex.stockpile.internal.api.TAssignShardResponse;
import ru.yandex.stockpile.internal.api.TListAssignmentsRequest;
import ru.yandex.stockpile.internal.api.TListAssignmentsResponse;
import ru.yandex.stockpile.internal.api.TPingRequest;
import ru.yandex.stockpile.internal.api.TPingResponse;
import ru.yandex.stockpile.internal.api.TUnassignShardRequest;
import ru.yandex.stockpile.internal.api.TUnassignShardResponse;


/**
 * @author Vladimir Gordiychuk
 */
public class GrpcStockpileBalancerClient implements StockpileBalancerClient {
    private final ClusterDiscovery<GrpcTransport> discovery;

    public GrpcStockpileBalancerClient(ClusterDiscovery<GrpcTransport> discovery) {
        this.discovery = discovery;
    }

    @Override
    public Set<String> getNodes() {
        return discovery.getNodes();
    }

    @Override
    public CompletableFuture<TAssignShardResponse> assignShard(String node, TAssignShardRequest request) {
        return unaryCall(node, StockpileBalancerServiceGrpc.getAssignShardMethod(), request, request.getExpiredAt());
    }

    @Override
    public CompletableFuture<TUnassignShardResponse> unassignShard(String node, TUnassignShardRequest request) {
        return unaryCall(node, StockpileBalancerServiceGrpc.getUnassignShardMethod(), request, request.getExpiredAt());
    }

    @Override
    public CompletableFuture<TListAssignmentsResponse> listAssignments(String leader, TListAssignmentsRequest request) {
        return unaryCall(leader, StockpileBalancerServiceGrpc.getListAssignmentsMethod(), request, request.getExpiredAt());
    }

    @Override
    public CompletableFuture<TPingResponse> ping(String node, TPingRequest request) {
        return unaryCall(node, StockpileBalancerServiceGrpc.getPingMethod(), request, request.getExpiredAt());
    }

    private <ReqT, RespT> CompletableFuture<RespT> unaryCall(String target, MethodDescriptor<ReqT, RespT> method, ReqT request, long expiredAt) {
        try {
            GrpcTransport transport = discovery.getTransportByNode(target);
            return transport.unaryCall(method, request, expiredAt);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }
}
