package ru.yandex.stockpile.cluster.balancer.dao;

import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntArrayMap;

import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.CommonResource;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;
import ru.yandex.solomon.balancer.snapshot.SnapshotNode;
import ru.yandex.solomon.balancer.snapshot.SnapshotShard;
import ru.yandex.solomon.protobuf.stockpile.TStockpileBalancerAssignmentState;
import ru.yandex.solomon.protobuf.stockpile.TStockpileBalancerNodeLimits;
import ru.yandex.solomon.protobuf.stockpile.TStockpileBalancerOptions;
import ru.yandex.solomon.protobuf.stockpile.TStockpileNode;
import ru.yandex.solomon.protobuf.stockpile.TStockpileShard;
import ru.yandex.stockpile.client.shard.StockpileShardId;

/**
 * @author Vladimir Gordiychuk
 */
class Converter {
    public static TStockpileBalancerAssignmentState toProto(SnapshotAssignments assignments) {
        var nodeToId = new Object2IntArrayMap<String>(assignments.nodes.size());
        TStockpileBalancerAssignmentState.Builder state = TStockpileBalancerAssignmentState.newBuilder();
        for (var node : assignments.nodes) {
            int id = state.getNodesCount() + 1;
            state.addNodes(TStockpileNode.newBuilder()
                    .setActive(node.active)
                    .setFreeze(node.freeze)
                    .setAddress(node.address)
                    .setId(id)
                    .build());
            nodeToId.put(node.address, id);
        }

        for (var shard : assignments.shards) {
            int nodeId = shard.node != null
                    ? nodeToId.getInt(shard.node)
                    : 0;

            state.addShards(TStockpileShard.newBuilder()
                    .setNodeId(nodeId)
                    .setShardId(StockpileShardId.parse(shard.shardId))
                    .setUtimeMillis(shard.resources.get(CommonResource.CPU))
                    .setMemoryBytes(Math.round(shard.resources.get(CommonResource.MEMORY)))
                    .setMetricsReadRate(shard.resources.get(CommonResource.METRICS_READ_RATE))
                    .setRecordsWriteRate(shard.resources.get(CommonResource.RECORDS_WRITE_RATE))
                    .build());
        }

        return state.build();
    }

    public static SnapshotAssignments fromProto(TStockpileBalancerAssignmentState proto) {
        var nodeAddressById = new Int2ObjectOpenHashMap<String>();
        var nodes = new ArrayList<SnapshotNode>(proto.getNodesCount());
        for (var nodeProto : proto.getNodesList()) {
            var node = new SnapshotNode();
            node.active = nodeProto.getActive();
            node.freeze = nodeProto.getFreeze();
            node.address = nodeProto.getAddress();
            nodeAddressById.put(nodeProto.getId(), node.address);
            nodes.add(node);
        }

        var shards = new ArrayList<SnapshotShard>(proto.getShardsCount());
        for (var shardProto : proto.getShardsList()) {
            var shard = new SnapshotShard();
            shard.shardId = StockpileShardId.toString(shardProto.getShardId());
            shard.node = nodeAddressById.get(shardProto.getNodeId());
            shard.resources = new Resources();
            shard.resources.set(CommonResource.CPU, shardProto.getUtimeMillis());
            shard.resources.set(CommonResource.MEMORY, shardProto.getMemoryBytes());
            shard.resources.set(CommonResource.METRICS_READ_RATE, shardProto.getMetricsReadRate());
            shard.resources.set(CommonResource.RECORDS_WRITE_RATE, shardProto.getRecordsWriteRate());
            shards.add(shard);
        }

        return new SnapshotAssignments(nodes, shards);
    }

    public static BalancerOptions fromProto(TStockpileBalancerOptions proto) {
        if (proto == null || TStockpileBalancerOptions.getDefaultInstance().equals(proto)) {
            return BalancerOptions.newBuilder().build();
        }
        TStockpileBalancerNodeLimits protoLimits = proto.getLimits();
        return BalancerOptions.newBuilder()
                .setVersion(proto.getVersion())
                .setAssignExpiration(proto.getAssignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setHeartbeatExpiration(proto.getHeartbeatExpirationMillis(), TimeUnit.MILLISECONDS)
                .setGracefulUnassignExpiration(proto.getGracefulUnassignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setForceUnassignExpiration(proto.getForceUnassignExpirationMillis(), TimeUnit.MILLISECONDS)
                .setAutoRebalanceDispersionThreshold(proto.getAutoReassignDispersionThreshold())
                .setLimits(fromProto(protoLimits))
                .setMaxReassignInFlight(Math.toIntExact(proto.getMaxReassignInFlight()))
                .setMaxLongLoadingShardsToIgnore(Math.toIntExact(proto.getMaxLongLoadingShardsToIgnore()))
                .setEnableAutoRebalance(proto.getEnableAutoRebalance())
                .setRebalanceDispersionTarget(proto.getRebalanceDispersionTarget())
                .setDisableAutoFreeze(proto.getDisableAutoFreeze())
                .build();
    }

    public static Resources fromProto(TStockpileBalancerNodeLimits proto) {
        if (proto == null || TStockpileBalancerNodeLimits.getDefaultInstance().equals(proto)) {
            return null;
        }

        Resources r = new Resources();
        r.set(CommonResource.CPU, proto.getUtimeMillis());
        r.set(CommonResource.MEMORY, proto.getMemoryBytes());
        r.set(CommonResource.SHARDS_COUNT, proto.getShardsCount());
        r.set(CommonResource.METRICS_READ_RATE, proto.getMetricsReadRate());
        r.set(CommonResource.RECORDS_WRITE_RATE, proto.getRecordsWriteRate());
        return r;
    }

    public static TStockpileBalancerOptions toProto(BalancerOptions opts) {
        return TStockpileBalancerOptions.newBuilder()
                .setVersion(opts.getVersion())
                .setAssignExpirationMillis(opts.getAssignExpirationMillis())
                .setHeartbeatExpirationMillis(opts.getHeartbeatExpirationMillis())
                .setGracefulUnassignExpirationMillis(opts.getGracefulUnassignExpirationMillis())
                .setForceUnassignExpirationMillis(opts.getForceUnassignExpirationMillis())
                .setAutoReassignDispersionThreshold(opts.getAutoRebalanceDispersionThreshold())
                .setLimits(toProto(opts.getLimits()))
                .setMaxReassignInFlight(opts.getMaxReassignInFlight())
                .setMaxLongLoadingShardsToIgnore(opts.getMaxLongLoadingShardsToIgnore())
                .setEnableAutoRebalance(opts.isEnableAutoRebalance())
                .setRebalanceDispersionTarget(opts.getRebalanceDispersionTarget())
                .setDisableAutoFreeze(opts.isDisableAutoFreeze())
                .build();
    }

    public static TStockpileBalancerNodeLimits toProto(Resources resources) {
        return TStockpileBalancerNodeLimits.newBuilder()
                .setUtimeMillis(resources.get(CommonResource.CPU))
                .setMemoryBytes(Math.round(resources.get(CommonResource.MEMORY)))
                .setShardsCount(Math.round(resources.get(CommonResource.SHARDS_COUNT)))
                .setMetricsReadRate(resources.get(CommonResource.METRICS_READ_RATE))
                .setRecordsWriteRate(resources.get(CommonResource.RECORDS_WRITE_RATE))
                .build();
    }
}
