package ru.yandex.stockpile.cluster.balancer.dao;

import java.util.concurrent.CompletableFuture;

import com.google.protobuf.InvalidProtocolBufferException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.BalancerOptions;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.snapshot.SnapshotAssignments;
import ru.yandex.solomon.protobuf.stockpile.TStockpileBalancerAssignmentState;
import ru.yandex.solomon.protobuf.stockpile.TStockpileBalancerOptions;
import ru.yandex.stockpile.kikimrKv.KvChannel;

/**
 * @author Vladimir Gordiychuk
 */
public class KvStockpileBalancerDao implements BalancerDao {
    private static final Logger logger = LoggerFactory.getLogger(KvStockpileBalancerDao.class);
    private static final String OPTIONS_FILE = "stockpile.balancer.options";
    private static final String ASSIGNMENT_STATE_FILE = "stockpile.balancer.state";

    private final KikimrKvClient kvClient;
    private final String path;
    private volatile long tabletId;

    public KvStockpileBalancerDao(KikimrKvClient kvClient, String path) {
        this(kvClient, path, 0);
    }

    public KvStockpileBalancerDao(KikimrKvClient kvClient, String path, long tabletId) {
        this.kvClient = kvClient;
        this.path = path;
        this.tabletId = tabletId;
    }

    @Override
    public CompletableFuture<BalancerOptions> getOptions() {
        return kvClient.readData(tabletId, 0, OPTIONS_FILE, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME)
            .thenApply(content -> {
                if (content.isEmpty() || content.get().length == 0) {
                    logger.info("absent persistence options for balancer, use defaults");
                    return BalancerOptions.newBuilder().build();
                }

                try {
                    var proto = TStockpileBalancerOptions.parseFrom(content.get());
                    return Converter.fromProto(proto);
                } catch (InvalidProtocolBufferException e) {
                    throw new RuntimeException(e);
                }
            });
    }

    @Override
    public CompletableFuture<SnapshotAssignments> getAssignments() {
        return kvClient.readData(tabletId, 0, ASSIGNMENT_STATE_FILE, 0, MsgbusKv.TKeyValueRequest.EPriority.REALTIME)
            .thenApply(content -> {
                if (content.isEmpty() || content.get().length == 0) {
                    return SnapshotAssignments.EMPTY;
                }

                try {
                    var proto = TStockpileBalancerAssignmentState.parseFrom(content.get());
                    return Converter.fromProto(proto);
                } catch (InvalidProtocolBufferException e) {
                    throw new RuntimeException(e);
                }
            });
    }

    @Override
    public CompletableFuture<Void> saveAssignments(SnapshotAssignments assignments) {
        var proto = Converter.toProto(assignments);
        return CompletableFutures.safeCall(() -> kvClient.write(
            tabletId,
            0,
            ASSIGNMENT_STATE_FILE,
            proto.toByteString(),
            KvChannel.SSD1.getChannel(),
            MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND, 0));
    }

    @Override
    public CompletableFuture<Void> saveOptions(BalancerOptions opts) {
        var proto = Converter.toProto(opts);
        return CompletableFutures.safeCall(() -> kvClient.write(
            tabletId,
            0,
            OPTIONS_FILE,
            proto.toByteString(),
            KvChannel.SSD1.getChannel(),
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME, 0));
    }

    @Override
    public CompletableFuture<Void> createSchema() {
        if (tabletId != 0) {
            return CompletableFuture.completedFuture(null);
        }

        return kvClient.createKvTablets(path, 1)
            .thenCompose(ignore -> kvClient.resolveKvTablets(path))
            .thenAccept(tabletIds -> {
                if (tabletIds.length != 1) {
                    throw new IllegalStateException("By path " + path + " should be only one tablet into KV");
                }

                tabletId = tabletIds[0];
            });
    }
}
