package ru.yandex.stockpile.cluster.balancer.leader;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.balancer.CommonResource;
import ru.yandex.solomon.balancer.Resources;
import ru.yandex.solomon.balancer.ShardStatus;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.remote.RemoteNodeState;
import ru.yandex.solomon.balancer.remote.RemoteShardState;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.cluster.balancer.KvTabletClient;
import ru.yandex.stockpile.cluster.balancer.client.StockpileBalancerClient;
import ru.yandex.stockpile.internal.api.TAssignShardRequest;
import ru.yandex.stockpile.internal.api.TPingRequest;
import ru.yandex.stockpile.internal.api.TPingResponse;
import ru.yandex.stockpile.internal.api.TShardAssignment;
import ru.yandex.stockpile.internal.api.TShardSummary;
import ru.yandex.stockpile.internal.api.TShardSummary.EStatus;
import ru.yandex.stockpile.internal.api.TUnassignShardRequest;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileRemoteNodeClient implements RemoteNodeClient {
    private final StockpileBalancerClient client;
    private final KvTabletClient kvTabletClient;

    public StockpileRemoteNodeClient(StockpileBalancerClient client, KvTabletClient kvTabletClient) {
        this.client = client;
        this.kvTabletClient = kvTabletClient;
    }

    @Override
    public boolean hasNode(String node) {
        return client.getNodes().contains(node);
    }

    @Override
    public Set<String> getNodes() {
        return client.getNodes();
    }

    @Override
    public CompletableFuture<Void> assignShard(String address, String shardId, AssignmentSeqNo seqNo, long expiredAt) {
        int id = StockpileShardId.parse(shardId);
        return kvTabletClient.incrementGeneration(id)
                .thenCompose(tablet -> {
                    var req = TAssignShardRequest.newBuilder()
                            .setLeaderSeqNo(seqNo.getLeaderSeqNo())
                            .setAssignment(TShardAssignment.newBuilder()
                                    .setShardId(id)
                                    .setTabletId(tablet.getTabletId())
                                    .setTabletGeneration(tablet.getGen())
                                    .build())
                            .setExpiredAt(expiredAt)
                            .build();

                    return client.assignShard(address, req);
                })
                .thenRun(() -> {});
    }

    @Override
    public CompletableFuture<Void> unassignShard(String address, String shardId, AssignmentSeqNo seqNo, boolean graceful, long expiredAt) {
        var req = TUnassignShardRequest.newBuilder()
                .setLeaderSeqNo(seqNo.getLeaderSeqNo())
                .setShardId(StockpileShardId.parse(shardId))
                .setGraceful(graceful)
                .setExpiredAt(expiredAt)
                .build();

        return client.unassignShard(address, req).thenRun(() -> {});
    }

    @Override
    public CompletableFuture<RemoteNodeState> ping(String address, long leaderSeqNo, long expiredAt) {
        var req = TPingRequest.newBuilder()
                .setLeaderSeqNo(leaderSeqNo)
                .setShardCount(kvTabletClient.getShardCount())
                .setExpiredAt(expiredAt)
                .build();
        return client.ping(address, req).thenApply(this::convert);
    }

    private RemoteNodeState convert(TPingResponse response) {
        RemoteNodeState result = new RemoteNodeState();
        result.uptimeMillis = response.getNodeSummary().getUpTimeMillis();
        result.memoryBytes = response.getNodeSummary().getMemoryBytes();
        result.utimeMillis = response.getNodeSummary().getUtimeMillis();
        result.shards = response.getShardSummaryList()
                .stream()
                .map(this::convert)
                .collect(Collectors.toList());
        result.receivedAt = System.currentTimeMillis();
        return result;
    }

    private RemoteShardState convert(TShardSummary summary) {
        var result = new RemoteShardState();
        result.shardId = Integer.toString(summary.getShardId());
        result.uptimeMillis = summary.getUpTimeMillis();
        result.status = toStatus(summary.getStatus());
        result.resources = new Resources();
        result.resources.set(CommonResource.CPU, summary.getUtimeMillis());
        result.resources.set(CommonResource.MEMORY, summary.getMemoryBytes());
        result.resources.set(CommonResource.METRICS_READ_RATE, summary.getMetricsReadRate());
        result.resources.set(CommonResource.RECORDS_WRITE_RATE, summary.getRecordsWriteRate());
        return result;
    }

    private static ShardStatus toStatus(EStatus state) {
        switch (state) {
            case READY:
                return ShardStatus.READY;
            case LOADING:
                return ShardStatus.LOADING;
            case INIT:
                return ShardStatus.INIT;
            default:
                return ShardStatus.UNKNOWN;
        }
    }
}
