package ru.yandex.stockpile.cluster.discovery;

import java.util.Optional;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.GrpcClusterDiscovery;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.stockpile.TStockpileDiscoveryConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.ctx.ServiceAuthContext;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
        GrpcConfigurationContext.class,
        ServiceAuthContext.class
})
public class StockpileDiscoveryContext {
    private final String clientId;
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;

    public StockpileDiscoveryContext(@Qualifier("ClientId") Optional<String> clientId, ThreadPoolProvider threads, MetricRegistry registry) {
        this.clientId = clientId.map(Strings::emptyToNull).orElse("stockpile-server");
        this.threads = threads;
        this.registry = registry;
    }

    @Bean
    public ClusterDiscovery<GrpcTransport> stockpileDiscovery(
            TStockpileDiscoveryConfig config,
            ClientOptionsFactory clientOptionsFactory)
    {
        var addresses = config.getInternalGrpcConfig().getAddressesList();
        var opts = clientOptionsFactory.newBuilder("", config.getInternalGrpcConfig())
            .setClientId(clientId)
            .setMetricRegistry(registry)
            .build();
        var timer = threads.getSchedulerExecutorService();
        var executor = threads.getExecutorService(config.getThreadPoolName(), "StockpileDiscoveryConfig.ThreadPoolName");
        var refreshMillis = TimeUnitConverter.millis(config.getRefreshInterval());
        return GrpcClusterDiscovery.of(opts, addresses, timer, executor, refreshMillis);
    }
}
