package ru.yandex.stockpile.kikimrKv;

import java.util.EnumMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;

import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.staffOnly.manager.table.TableColumnDefImpl;
import ru.yandex.stockpile.server.data.names.FileKind;

/**
 * SSD/HDD write policy.
 *
 * @author Stepan Koltsov
 */
@LinkedOnRootPage("Channels mapping")
@ParametersAreNonnullByDefault
public class KvChannels {
    private static final int INLINE_WRITE_BYTES_THRESHOLD = 1024;

    public static final MsgbusKv.TKeyValueRequest.EPriority logPriority =
        MsgbusKv.TKeyValueRequest.EPriority.REALTIME;
    public static final MsgbusKv.TKeyValueRequest.EPriority snapshotPriority =
        MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND;

    private static volatile EnumMap<FileKind, KvChannel> kindToChannel;
    @Nonnull
    private static volatile KvChannel inlineFallback = KvChannel.SSD1;

    static {
        EnumMap<FileKind, KvChannel> channels = Maps.newEnumMap(FileKind.class);
        channels.put(FileKind.LOG, KvChannel.INLINE);
        channels.put(FileKind.LOG_SNAPSHOT, KvChannel.SSD1);
        channels.put(FileKind.ETERNITY_CHUNK, KvChannel.HDD);
        channels.put(FileKind.ETERNITY_INDEX, KvChannel.HDD);
        channels.put(FileKind.ETERNITY_COMMAND, KvChannel.HDD);
        channels.put(FileKind.DAILY_CHUNK, KvChannel.HDD);
        channels.put(FileKind.DAILY_INDEX, KvChannel.HDD);
        channels.put(FileKind.DAILY_COMMAND, KvChannel.HDD);
        channels.put(FileKind.TWO_HOUR_CHUNK, KvChannel.SSD1);
        channels.put(FileKind.TWO_HOUR_INDEX, KvChannel.SSD1);
        channels.put(FileKind.TWO_HOUR_COMMAND, KvChannel.SSD1);
        channels.put(FileKind.PRODUCER_SEQUENCES, KvChannel.SSD1);
        kindToChannel = channels;
    }

    public static KvChannel byFileKind(FileKind fileKind) {
        return kindToChannel.getOrDefault(fileKind, KvChannel.HDD);
    }

    public static KvChannel byFileKind(FileKind fileKind, int bytes) {
        var channel = byFileKind(fileKind);
        if (channel == KvChannel.INLINE && bytes >= INLINE_WRITE_BYTES_THRESHOLD) {
            return inlineFallback;
        }
        return channel;
    }

    @ManagerMethod
    public void changeChannel(FileKind fileKind, KvChannel channel) {
        EnumMap<FileKind, KvChannel> copy = Maps.newEnumMap(kindToChannel);
        copy.put(fileKind, channel);
        kindToChannel = copy;
    }

    @ManagerMethod
    public void changeInlineFallback(KvChannel channel) {
        inlineFallback = channel;
    }

    @ExtraContent("Mapping")
    public void extra(ExtraContentParam p) {
        Map<FileKind, KvChannel> copy = Maps.newEnumMap(kindToChannel);
        var fallback = inlineFallback;
        p.managerWriter().listTable(copy.entrySet(),
            new TableColumnDefImpl<>("File kind", i -> i.getKey().name()),
            new TableColumnDefImpl<>("Channel", i -> {
                var name = i.getValue().name();
                if (i.getValue() != KvChannel.INLINE) {
                    return name;
                }

                return name + "(" + fallback +")";
            }),
            new TableColumnDefImpl<>("Storage", i -> {
                var name = i.getValue().getChannel().name();
                if (i.getValue() != KvChannel.INLINE) {
                    return name;
                }
                return name + "(" + fallback.getChannel().name() + ")";
            })
        );
    }
}
