package ru.yandex.stockpile.kikimrKv;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryStats;
import ru.yandex.kikimr.client.kv.KikimrKvClient.Rename;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.ChunkFile;
import ru.yandex.stockpile.server.data.names.file.CommandFile;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */

@LinkedOnRootPage("KV files")
@ParametersAreNonnullByDefault
public class KvFiles {
    private static final String DELETE_PREFIX = "d.";

    private final KikimrKvClient client;
    private final String volumePath;

    public KvFiles(KikimrKvClient client, String volumePath) {
        this.client = client;
        this.volumePath = volumePath;
    }

    @ManagerMethod
    public CompletableFuture<Void> deleteSnapshot(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "level") SnapshotLevel level,
            @ManagerMethodArgument(name = "txn") long txn)
    {
        var address = new SnapshotAddress(level, txn);
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var ranges = StockpileKvNames.currentSnapshot(address);
                    return addPrefix(tabletId, DELETE_PREFIX, ranges)
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> restoreSnapshot(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "level") SnapshotLevel level,
            @ManagerMethodArgument(name = "txn") long txn)
    {
        var address = new SnapshotAddress(level, txn);
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var ranges = StockpileKvNames.currentSnapshot(DELETE_PREFIX, address);
                    return removePrefix(tabletId, DELETE_PREFIX, ranges)
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> deleteLog(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "txn") long txn)
    {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(LogFile.pfCurrentWithoutPart.format(txn));
                    return addPrefix(tabletId, DELETE_PREFIX, List.of(range))
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> restoreLog(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "txn") long txn)
    {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(DELETE_PREFIX + LogFile.pfCurrentWithoutPart.format(txn));
                    return removePrefix(tabletId, DELETE_PREFIX, List.of(range))
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> deleteUnknownFile(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "file") String name)
    {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(name);
                    return addPrefix(tabletId, DELETE_PREFIX, List.of(range))
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> restoreUnknownFile(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "file") String name)
    {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(name);
                    return removePrefix(tabletId, DELETE_PREFIX, List.of(range))
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> deleteFile(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "fileName") String file)
    {
        var parsed = FileNameParsed.parseCurrent(file);
        if (parsed instanceof IndexFile index) {
            return deleteSnapshot(shardId, index.getLevel(), index.txn());
        } else if (parsed instanceof ChunkFile chunk) {
            var address = chunk.snapshotAddress();
            return deleteSnapshot(shardId, address.level(), address.txn());
        } else if (parsed instanceof CommandFile command) {
            var address = command.snapshotAddress();
            return deleteSnapshot(shardId, address.level(), address.txn());
        } else if (parsed instanceof LogFile log) {
            return deleteLog(shardId, log.txn());
        } else {
            return deleteUnknownFile(shardId, file);
        }
    }

    @ManagerMethod
    public CompletableFuture<Void> restoreFile(
            @ManagerMethodArgument(name = "shardId") int shardId,
            @ManagerMethodArgument(name = "fileName") String file)
    {
        var parsed = FileNameParsed.parseCurrent(file);
        if (parsed instanceof IndexFile index) {
            return restoreSnapshot(shardId, index.getLevel(), index.txn());
        } else if (parsed instanceof ChunkFile chunk) {
            var address = chunk.snapshotAddress();
            return restoreSnapshot(shardId, address.level(), address.txn());
        } else if (parsed instanceof CommandFile command) {
            var address = command.snapshotAddress();
            return restoreSnapshot(shardId, address.level(), address.txn());
        } else if (parsed instanceof LogFile log) {
            return restoreLog(shardId, log.txn());
        } else {
            return restoreUnknownFile(shardId, file);
        }
    }

    @ManagerMethod
    public CompletableFuture<Void> cleanupRestoreFiles(@ManagerMethodArgument(name = "shardId") int shardId) {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(DELETE_PREFIX);
                    return client.deleteRange(tabletId, 0, range, 0);
                });
    }

    @ManagerMethod
    public CompletableFuture<Void> restoreAllDeleted(@ManagerMethodArgument(name = "shardId") int shardId) {
        return tabletIdByShardId(shardId)
                .thenCompose(tabletId -> {
                    var range = StringMicroUtils.asciiPrefixToRange(DELETE_PREFIX);
                    return removePrefix(tabletId, DELETE_PREFIX, List.of(range))
                            .thenCompose(ignore2 -> incGeneration(tabletId))
                            .thenRun(() -> {});
                });
    }

    public CompletableFuture<Long> tabletIdByShardId(int shardId) {
        return client.resolveKvTablets(volumePath)
                .thenApply(tabletIds -> {
                    var map = new ShardIdMapToLong(tabletIds);
                    return map.get(shardId);
                });
    }

    private CompletableFuture<Void> addPrefix(long tabletId, String prefix, List<NameRange> ranges) {
        return listFiles(tabletId, ranges)
                .thenCompose(files -> {
                    var renames = files.stream()
                            .map(file -> new Rename(file.getName(), prefix + file.getName()))
                            .collect(Collectors.toList());

                    return client.renameMultiple(tabletId, 0, renames, 0);
                });
    }

    private CompletableFuture<Void> removePrefix(long tabletId, String prefix, List<NameRange> ranges) {
        return listFiles(tabletId, ranges)
                .thenCompose(files -> {
                    var renames = files.stream()
                            .filter(file -> file.getName().startsWith(prefix))
                            .map(file -> new Rename(file.getName(), file.getName().substring(prefix.length())))
                            .collect(Collectors.toList());

                    return client.renameMultiple(tabletId, 0, renames, 0);
                });
    }

    private CompletableFuture<Long> incGeneration(long tabletId) {
        return client.incrementGeneration(tabletId, 0);
    }

    private CompletableFuture<List<KvEntryStats>> listFiles(long tabletId, List<NameRange> ranges) {
        return ranges.stream()
                .map(range -> listFiles(tabletId, range))
                .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(lists -> lists.stream()
                        .flatMap(Collection::stream)
                        .collect(toList()));
    }

    private CompletableFuture<List<KvEntryStats>> listFiles(long tabletId, NameRange range) {
        return client.readRangeNames(tabletId, 0, range, 0);
    }

}
