package ru.yandex.stockpile.kikimrKv;

import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryStats;
import ru.yandex.kikimr.util.NameRange;

/**
 * @author Vladimir Gordiychuk
 */
public class KvListFiles {
    private final KikimrKvClient kvClient;
    private final long tabletId;
    private final long gen;
    private final Clock clock;
    private final long timeout;

    public KvListFiles(KikimrKvClient kvClient, long tabletId, long gen, Clock clock, long timeout) {
        this.kvClient = kvClient;
        this.tabletId = tabletId;
        this.gen = gen;
        this.clock = clock;
        this.timeout = timeout;
    }

    public CompletableFuture<List<KvEntryStats>> listFiles(NameRange range) {
        var doneFuture = new CompletableFuture<List<KvEntryStats>>();
        var result = new ArrayList<KvEntryStats>();
        listFiles(range, result, doneFuture);
        return doneFuture;
    }

    private void listFiles(NameRange range, ArrayList<KvEntryStats> result, CompletableFuture<List<KvEntryStats>> doneFuture) {
        kvClient.readRange(tabletId, gen, range, false, expiredAt())
                .whenComplete((response, e) -> {
                    if (e != null) {
                        doneFuture.completeExceptionally(e);
                        return;
                    }

                    try {
                        var entries = response.getEntries();
                        result.ensureCapacity(result.size() + entries.length);
                        for (var entry : response.getEntries()) {
                            var stats = entry.toStats();
                            result.add(stats);
                        }

                        if (!response.isOverrun()) {
                            doneFuture.complete(result);
                            return;
                        }

                        var nextRange = NameRange.rangeFrom(response.getOverrunLastEntryName(), false);
                        if (range.getEnd() != null) {
                            nextRange = nextRange.to(range.getEnd(), range.isEndInclusive());
                        }
                        listFiles(nextRange, result, doneFuture);
                    } catch (Throwable e2) {
                        doneFuture.completeExceptionally(e2);
                    }
                });
    }

    private long expiredAt() {
        if (timeout == 0) {
            return 0;
        }

        return clock.millis() + timeout;
    }
}
