package ru.yandex.stockpile.kikimrKv;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient.Rename;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.naked.NakedDeserializer;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.ListOfOptional;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.chunk.ChunkAddressGlobal;
import ru.yandex.stockpile.server.data.chunk.ChunkWithNo;
import ru.yandex.stockpile.server.data.chunk.DataRangeGlobal;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.command.LongFileTypeSnapshotCommand;
import ru.yandex.stockpile.server.data.command.SnapshotCommandPartsSerialized;
import ru.yandex.stockpile.server.data.dao.LargeFileWriter;
import ru.yandex.stockpile.server.data.dao.ReadBatcher;
import ru.yandex.stockpile.server.data.dao.StockpileShardStorage;
import ru.yandex.stockpile.server.data.index.LongFileTypeSnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.index.SnapshotIndexPartsSerialized;
import ru.yandex.stockpile.server.data.log.LongFileTypeLogEntry;
import ru.yandex.stockpile.server.data.log.StockpileLogEntrySerialized;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.ProducerSeqNoFile;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;


/**
 * @author Sergey Polovko
 */
public class KvStockpileShardStorage implements StockpileShardStorage {
    public static final long TABLE_GENERATION_UNKNOWN = -1;

    private final ReadBatcher readBatcher;
    private final KikimrKvClientCounting kvClient;
    private final long tabletId;
    private long tabletGen;

    public KvStockpileShardStorage(
        ReadBatcher readBatcher,
        KikimrKvClientCounting kvClient,
        long tabletId) {
        this(readBatcher, kvClient, tabletId, TABLE_GENERATION_UNKNOWN);
    }

    public KvStockpileShardStorage(
        ReadBatcher readBatcher,
        KikimrKvClientCounting kvClient,
        long tabletId,
        long tableGen)
    {
        this.readBatcher = readBatcher;
        this.kvClient = kvClient;
        this.tabletId = tabletId;
        this.tabletGen = tableGen;
    }

    @Override
    public long getGeneration() {
        return tabletGen;
    }

    @Override
    public CompletableFuture<Void> lock() {
        if (tabletGen != TABLE_GENERATION_UNKNOWN) {
            return CompletableFuture.completedFuture(null);
        }

        return kvClient.incrementGeneration(tabletId)
            .thenAccept(gen -> tabletGen = gen);
        // NOTE: read of tabletGen is implicitly synchronized through returned future, because
        //       all threads will join or wait on it
    }

    @Override
    public CompletableFuture<Void> writeLogEntry(StockpileLogEntrySerialized serialized) {
        List<KikimrKvClient.Write> writes = LongFileTypeLogEntry.I.makeWrites(
            serialized.split(),
            KvChannels.byFileKind(FileKind.LOG, serialized.size()).getChannel(),
            KvChannels.logPriority);
        return writeWithTempAndDelete(WriteClass.LOG, writes, Collections.emptyList());
    }

    @Override
    public CompletableFuture<Void> writeLogSnapshot(long firstLogTxn, StockpileLogEntrySerialized serialized) {
        List<NameRange> deletes = (firstLogTxn != serialized.getTxn())
            ? Collections.singletonList(StockpileKvNames.logNameRangeInclusive(firstLogTxn, serialized.getTxn() - 1))
            : Collections.emptyList();

        List<KikimrKvClient.Write> writes = LongFileTypeLogEntry.I.makeWrites(
            serialized.split(),
            KvChannels.byFileKind(FileKind.LOG_SNAPSHOT, serialized.size()).getChannel(),
            KvChannels.logPriority);
        return writeWithTempAndDelete(WriteClass.LOG_SNAPSHOT, writes, deletes);
    }

    private CompletableFuture<Void> writeWithTempAndDelete(
        WriteClass writeClass,
        List<KikimrKvClient.Write> writes,
        List<NameRange> deletes)
    {
        if (writes.size() == 1) {
            return kvClient.writeAndRenameAndDelete(
                writeClass, tabletId, tabletGen, writes, Collections.emptyList(), deletes);
        }

        try {
            ArrayList<Rename> renames = new ArrayList<>(writes.size());
            ArrayList<CompletableFuture<Void>> writeFutures = new ArrayList<>(writes.size());

            for (KikimrKvClient.Write write : writes) {
                String tempName = StockpileKvNames.TMP_PREFIX + write.getName();
                renames.add(new Rename(tempName, write.getName()));

                writeFutures.add(kvClient.write(
                    writeClass, tabletId, tabletGen, tempName,
                    write.getValue(), write.getStorageChannel(), write.getPriority()));
            }

            return CompletableFutures.allOfVoid(writeFutures)
                .thenCompose(unit -> kvClient.writeAndRenameAndDelete(
                    writeClass, tabletId, tabletGen,
                    Collections.emptyList(), renames, deletes));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> writeSnapshotChunkToTemp(
        SnapshotLevel level, long txn, ChunkWithNo chunkWithNo)
    {
        String chunkFileName = StockpileKvNames.chunkFileName(level, txn, chunkWithNo.getNo(), FileNamePrefix.Current.instance);
        String chunkFileNameTmp = StockpileKvNames.TMP_PREFIX + chunkFileName;
        MsgbusKv.TKeyValueRequest.EStorageChannel ssd = KvChannels.byFileKind(level.chunk.kind()).getChannel();

        return kvWriteLarge(
            level.chunk.writeClass(), chunkFileNameTmp, chunkWithNo.getContent(), ssd, KvChannels.snapshotPriority);
    }

    private CompletableFuture<Void> kvWriteLarge(
        WriteClass writeClass,
        String key, byte[] value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel,
        MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        if (value.length < KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE) {
            return kvWrite(writeClass, key, value, storageChannel, priority);
        }

        LargeFileWriter writer = new LargeFileWriter(kvClient, tabletId, tabletGen, writeClass, storageChannel, priority);
        return writer.write(key, value);
    }

    private CompletableFuture<Void> kvWrite(
        WriteClass writeClass,
        String key, byte[] value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel,
        MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        ByteString content = ByteStringsStockpile.unsafeWrap(value);
        return kvClient.writeAndRenameAndDeleteAndConcat(
            writeClass,
            tabletId, tabletGen,
            Collections.singletonList(new KikimrKvClient.Write(key, content, storageChannel, priority)),
            Collections.emptyList(),
            Collections.emptyList(),
            Collections.emptyList());
    }

    @Override
    public CompletableFuture<Void> writeSnapshotIndexToTemp(SnapshotIndexPartsSerialized parts) {
        MsgbusKv.TKeyValueRequest.EStorageChannel ssd = KvChannels.byFileKind(parts.getLevel().index.kind()).getChannel();
        List<KikimrKvClient.Write> writes = LongFileTypeSnapshotIndex.I.makeWrites(parts, ssd, KvChannels.snapshotPriority);
        List<CompletableFuture<Void>> writeFutures = new ArrayList<>();

        for (KikimrKvClient.Write write : writes) {
            String tmpName = StockpileKvNames.TMP_PREFIX + write.getName();
            writeFutures.add(kvClient.write(
                parts.getLevel().index.writeClass(),
                tabletId,
                tabletGen,
                tmpName,
                write.getValue(),
                ssd,
                KvChannels.snapshotPriority));
        }

        return CompletableFutures.allOfVoid(writeFutures);
    }

    @Override
    public CompletableFuture<Void> writeSnapshotCommandToTemp(SnapshotCommandPartsSerialized parts) {
        var command = parts.level().command;
        var channel = KvChannels.byFileKind(command.kind()).getChannel();
        var writes = LongFileTypeSnapshotCommand.I.makeWrites(parts, channel, KvChannels.snapshotPriority);

        var future = CompletableFuture.<Void>completedFuture(null);
        for (var write : writes) {
            future = future.thenCompose(ignore -> {
                var name = StockpileKvNames.TMP_PREFIX + write.getName();
                return kvClient.write(command.writeClass(), tabletId, tabletGen, name, write.getValue(), write.getStorageChannel(), write.getPriority());
            });
        }
        return future;
    }

    @Override
    public CompletableFuture<Void> writeProducerSeqNoSnapshot(byte[] snapshot) {
        var channel = KvChannels.byFileKind(FileKind.PRODUCER_SEQUENCES).getChannel();
        var priority = KvChannels.snapshotPriority;
        return kvWriteLarge(WriteClass.OTHER, ProducerSeqNoFile.CURRENT_FILE_NAME, snapshot, channel, priority);
    }

    @Override
    public CompletableFuture<Void> deleteTempFiles() {
        NameRange tmpRange = StringMicroUtils.asciiPrefixToRange(StockpileKvNames.TMP_PREFIX);
        return kvClient.deleteRange(WriteClass.OTHER, tabletId, tabletGen, tmpRange);
    }

    @Override
    public CompletableFuture<Void> renameSnapshotDeleteLogs(SnapshotAddress address) {
        return listTmpSnapshot(address)
            .thenCompose(renames -> {
                NameRange delete = StockpileKvNames.logsToRange(address.txn(), true);
                return kvClient.writeAndRenameAndDelete(
                    WriteClass.META,
                    tabletId, tabletGen,
                    Collections.emptyList(),
                    renames,
                    Collections.singletonList(delete));
            });
    }

    @Override
    public CompletableFuture<Void> renameSnapshotDeleteOld(
        SnapshotAddress[] renameSnapshots, SnapshotAddress[] deleteSnapshots)
    {
        var minTxn = Stream.of(renameSnapshots)
                .min(SnapshotAddress::compareTo)
                .map(SnapshotAddress::txn)
                .orElse(Long.MAX_VALUE);

        return listTmpSnapshot(renameSnapshots)
            .thenCompose(renames -> {
                ArrayList<NameRange> deletes = new ArrayList<>();
                for (SnapshotAddress delete : deleteSnapshots) {
                    Txn.validateTxn(delete.txn());
                    if (delete.txn() >= minTxn) {
                        throw new IllegalArgumentException("Delete txn " + delete.txn() + " >= " + minTxn);
                    }

                    deletes.addAll(StockpileKvNames.currentSnapshot(delete));
                }

                return kvClient.writeAndRenameAndDelete(
                    WriteClass.META,
                    tabletId, tabletGen,
                    Collections.emptyList(),
                    renames,
                    deletes);
            });
    }

    private CompletableFuture<List<Rename>> listTmpSnapshot(SnapshotAddress[] addresses) {
        return Stream.of(addresses)
            .map(this::listTmpSnapshot)
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(renames -> renames.stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toList()));
    }

    private CompletableFuture<List<Rename>> listTmpSnapshot(SnapshotAddress address) {
        return StockpileKvNames.currentSnapshot(StockpileKvNames.TMP_PREFIX, address)
                .stream()
                .map(nameRange -> readRangeNames(ReadClass.OTHER, nameRange))
                .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(lists -> lists.stream()
                        .flatMap(Collection::stream)
                        .map(file -> new Rename(file.getName(), file.getName().substring(StockpileKvNames.TMP_PREFIX.length())))
                        .collect(toList()));
    }

    @Override
    public CompletableFuture<byte[]> readChunk(
        ReadClass readClass, ChunkAddressGlobal chunkAddress, FileNamePrefix prefix)
    {
        String key = StockpileKvNames.chunkFileName(
            chunkAddress.getLevel(),
            chunkAddress.getSnapshotTxn(),
            chunkAddress.getChunkNo(),
            prefix);
        return kvClient.readDataLarge(readClass, tabletId, tabletGen, key);
    }

    @Override
    public CompletableFuture<List<KikimrKvClient.KvEntryStats>> readRangeNames(ReadClass readClass, NameRange nameRange) {
        return kvClient.readRangeNames(readClass, tabletId, tabletGen, nameRange);
    }

    @Override
    public CompletableFuture<Optional<byte[]>> readData(ReadClass readClass, String name) {
        return kvClient.readData(readClass, tabletId, tabletGen, name);
    }

    @Override
    public CompletableFuture<ListOfOptional<MetricArchiveImmutable>> readSnapshotRanges(DataRangeGlobal[] ranges) {
        if (ranges.length == 0) {
            return CompletableFuture.completedFuture(new ListOfOptional<>(new MetricArchiveImmutable[0]));
        }

        KvChunkAddress[] chunkAddresses = Arrays.stream(ranges)
            .map(KvStockpileShardStorage::chunkAddressGlobalToAddress)
            .toArray(KvChunkAddress[]::new);

        return readBatcher.kvReadDataMulti(tabletId, tabletGen, chunkAddresses)
            .thenApply(bytess -> {
                return IntStream.range(0, bytess.length).mapToObj(i -> {
                    byte[] bytes = bytess[i];
                    if (bytes.length == 0) {
                        return null;
                    } else {
                        StockpileFormat indexFormat = ranges[i].getIndexFormat();
                        NakedDeserializer<MetricArchiveImmutable> deserializer =
                            SnapshotIndexContentSerializer.dataDeserializerForVersionSealed(indexFormat);
                        return deserializer.deserializeFull(bytes);
                    }
                }).collect(ListOfOptional.collectorOfNullables(MetricArchiveImmutable[]::new));
            });
    }

    private static KvChunkAddress chunkAddressGlobalToAddress(DataRangeGlobal a) {
        String chunkFileName = StockpileKvNames.chunkFileName(
            a.getLevel(), a.getSnapshotTxn(), a.getChunkNo(), FileNamePrefix.Current.instance);
        return new KvChunkAddress(chunkFileName, a.getOffset(), a.getLength());
    }

    @Override
    public CompletableFuture<Optional<MetricArchiveImmutable>> readSnapshotRange(DataRangeGlobal range) {
        return readSnapshotRanges(new DataRangeGlobal[] { range })
            .thenApply(ListOfOptional::single);
    }

    @Override
    public CompletableFuture<Void> flushReadQueue() {
        return readBatcher.flushQueueForTablet(tabletId);
    }

    @Override
    public CompletableFuture<KvReadRangeResult> readRange(ReadClass readClass, NameRange nameRange) {
        return kvClient.readRangeData(readClass, tabletId, tabletGen, nameRange);
    }
}
