package ru.yandex.stockpile.kikimrKv;

import java.time.Clock;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.IntStream;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Sergey Polovko
 */
public class KvTabletsMapping implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(KvTabletsMapping.class);
    public static final long UNKNOWN_TABLET_ID = 0;
    private final long EXPIRE_TIME_MILLIS = TimeUnit.MINUTES.toMillis(10L);

    private final String kvVolumePath;
    private final KikimrKvClient kvClient;
    private final Clock clock;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    @Nullable
    private volatile ScheduledFuture<?> scheduled;
    private final CompletableFuture<?> ready = new CompletableFuture<>();
    private final AtomicReference<CompletableFuture<Void>> forceReload = new AtomicReference<>();

    private final ActorWithFutureRunner actor;
    private volatile Snapshot snapshot;
    private volatile boolean closed;

    public KvTabletsMapping(
        String kvVolumePath,
        KikimrKvClient kvClient,
        ExecutorService executor,
        ScheduledExecutorService timer)
    {
        this(kvVolumePath, kvClient, Clock.systemUTC(), executor, timer);
    }

    public KvTabletsMapping(
            String kvVolumePath,
            KikimrKvClient kvClient,
            Clock clock,
            ExecutorService executor,
            ScheduledExecutorService timer)
    {
        this.kvVolumePath = kvVolumePath;
        this.kvClient = kvClient;
        this.clock = clock;
        this.executor = executor;
        this.timer = timer;
        this.actor = new ActorWithFutureRunner(this::act, executor);
        actor.schedule();
    }

    private CompletableFuture<Void> act() {
        if (closed) {
            return CompletableFuture.completedFuture(null);
        }

        return resolveKvTablets()
                .thenAccept(tabletIds -> {
                    var actual = prepareSnapshot(tabletIds);
                    updateSnapshot(actual);
                    notifyAboutUpdate();
                })
                .exceptionally(e -> {
                    if (closed) {
                        return null;
                    }
                    logger.warn("cannot resolve KV tablets", e);
                    scheduled = timer.schedule(actor::schedule, 5, TimeUnit.SECONDS);
                    return null;
                });
    }

    private void notifyAboutUpdate() {
        var reloadFuture = forceReload.getAndSet(null);
        if (reloadFuture != null) {
            reloadFuture.completeAsync(() -> null, executor);
        }
        ready.completeAsync(() -> null, executor);
    }

    private CompletableFuture<long[]> resolveKvTablets() {
        return kvClient.resolveKvTablets(kvVolumePath)
            .thenApply(tabletIds -> {
                logger.info("{} tablets resolved with SchemeShard", tabletIds.length);
                return tabletIds;
            });
    }

    public CompletableFuture<Void> forceReload() {
        CompletableFuture<Void> future = new CompletableFuture<>();
        CompletableFuture<Void> prev = forceReload.get();
        do {
            if (prev != null && !prev.isDone()) {
                return prev;
            }
        } while (forceReload.compareAndSet(prev, future));
        actor.schedule();
        return future;
    }

    private Snapshot prepareSnapshot(long[] tabletIds) {
        if (new LongOpenHashSet(tabletIds).size() != tabletIds.length) {
            throw new IllegalArgumentException("non-unique tablet ids: " + Arrays.toString(tabletIds));
        }

        if (Arrays.stream(tabletIds).anyMatch(l -> l == UNKNOWN_TABLET_ID)) {
            throw new IllegalArgumentException("zero tablet id is invalid: " + Arrays.toString(tabletIds));
        }

        if (tabletIds.length == 0) {
            throw new IllegalArgumentException("resolved zero tables by kv volume: " + kvVolumePath);
        }

        var tabletIdByShardId = new ShardIdMapToLong(tabletIds);
        var tabletIdsSet = new LongOpenHashSet(tabletIds);
        return new Snapshot(tabletIdByShardId, tabletIdsSet, clock.millis());
    }

    private void updateSnapshot(Snapshot actual) {
        var prev = this.snapshot;
        if (prev != null) {
            if (!actual.tabletIdsSet.containsAll(prev.tabletIdsSet)) {
                throw new IllegalStateException("Absent some table ids, prev resolve " + prev.tabletIdsSet + " actual " + actual.tabletIdsSet);
            }

            for (int shardId = 1; shardId <= prev.tabletIdsSet.size(); shardId++) {
                var prevTabletId = prev.tabletIdByShardId.get(shardId);
                var actualTabletId = actual.tabletIdByShardId.get(shardId);
                if (prevTabletId != actualTabletId) {
                    throw new IllegalStateException(
                            "Tablet id changed " + prevTabletId + " -> " + actualTabletId
                                    + " for shardId " + StockpileShardId.toString(shardId));
                }
            }
        }
        this.snapshot = actual;
    }

    public void waitForReady() {
        ready.join();
    }

    public CompletableFuture<?> getReadyFuture() {
        return ready;
    }

    public boolean hasTabletId(long tabletId) {
        return snapshot().tabletIdsSet.contains(tabletId);
    }

    /**
     * @return tabletId or {@link KvTabletsMapping#UNKNOWN_TABLET_ID}
     */
    public long getTabletId(int shardId) {
        return snapshot().tabletIdByShardId.get(shardId);
    }

    public IntStream getShardIdStream() {
        return snapshot().tabletIdByShardId.shardIdStream();
    }

    public int getShardCount() {
        return snapshot().tabletIdsSet.size();
    }

    private Snapshot snapshot() {
        var copy = snapshot;
        if (copy == null) {
            throw new IllegalStateException("mapping not resolved yet");
        }

        if (copy.isMarkedExpired()) {
            return copy;

        }

        long expiredAt = copy.createdAt + EXPIRE_TIME_MILLIS;
        if (clock.millis() < expiredAt) {
            return copy;
        }

        if (copy.markExpired()) {
            actor.schedule();
        }

        return copy;
    }

    @Override
    public void close() {
        closed = true;
        var copy = scheduled;
        if (copy != null) {
            copy.cancel(false);
        }
    }

    private static class Snapshot {
        private final ShardIdMapToLong tabletIdByShardId;
        private final LongOpenHashSet tabletIdsSet;
        private final long createdAt;
        private final AtomicBoolean expired = new AtomicBoolean();

        public Snapshot(ShardIdMapToLong tabletIdByShardId, LongOpenHashSet tabletIdsSet, long createdAt) {
            this.tabletIdByShardId = tabletIdByShardId;
            this.tabletIdsSet = tabletIdsSet;
            this.createdAt = createdAt;
        }

        public boolean isMarkedExpired() {
            return expired.get();
        }

        public boolean markExpired() {
            return expired.compareAndSet(false, true);
        }
    }
}
