package ru.yandex.stockpile.kikimrKv.counting;

import java.time.Clock;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.ToLongFunction;

import javax.annotation.CheckReturnValue;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;
import ru.yandex.stockpile.kikimrKv.KvListFiles;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics.ReadMetrics;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics.WriteMetrics;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class KikimrKvClientCounting {

    private final KikimrKvClient kikimrKvClient;
    private final KikimrKvClientMetrics metrics;
    private final Clock clock;
    private final EnumMapToLong<ReadClass> readClassTimeouts;
    private final EnumMapToLong<WriteClass> writeClassTimeouts;

    public KikimrKvClientCounting(KikimrKvClient kvClient, KikimrKvClientMetrics metrics) {
        this(kvClient, metrics, Clock.systemUTC(), new EnumMapToLong<>(ReadClass.class), new EnumMapToLong<>(WriteClass.class));
    }

    public KikimrKvClientCounting(
        KikimrKvClient kikimrKvClient,
        KikimrKvClientMetrics metrics,
        Clock clock,
        EnumMapToLong<ReadClass> readClassTimeouts,
        EnumMapToLong<WriteClass> writeClassTimeouts)
    {
        this.kikimrKvClient = kikimrKvClient;
        this.metrics = metrics;
        this.clock = clock;
        this.readClassTimeouts = readClassTimeouts;
        this.writeClassTimeouts = writeClassTimeouts;
    }

    @CheckReturnValue
    public CompletableFuture<Void> writeAndRenameAndDeleteAndConcat(
        WriteClass writeClass,
        long tabletId, long gen,
        List<KikimrKvClient.Write> writes,
        List<KikimrKvClient.Rename> renames,
        List<NameRange> deletes,
        List<KikimrKvClient.Concat> concats)
    {
        WriteMetrics writeMetrics = metrics.getWriteMetrics(writeClass);
        final int ops = writes.size() + renames.size() + deletes.size() + concats.size();
        long bytes = 0;
        for (var write : writes) {
            bytes += write.getValue().size();
            writeMetrics.callByChannel(write.getStorageChannel());
        }
        writeMetrics.outboundBytes.add(bytes);
        writeMetrics.async.callStarted(ops);
        long startTimeNanos = System.nanoTime();
        return kikimrKvClient.writeAndRenameAndDeleteAndConcat(tabletId, gen, writes, renames, deletes, concats, expiredAt(writeClass))
            .whenComplete((r, e) -> {
                long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
                if (e != null) {
                    writeMetrics.callCompletedError(elapsedMillis, ops, e);
                    return;
                }

                writeMetrics.async.callCompletedOk(elapsedMillis, ops);
            });
    }

    @CheckReturnValue
    public CompletableFuture<KvReadRangeResult> readRangeData(
        ReadClass readClass,
        long tabletId, long gen, NameRange nameRange)
    {
        var future = kikimrKvClient.readRangeData(tabletId, gen, nameRange, expiredAt(readClass));
        measureAsyncReadOp(future, readClass, 1, KvReadRangeResult::bytesSize);
        return future;
    }

    @CheckReturnValue
    public CompletableFuture<Void> write(
        WriteClass writeClass, long tabletId, long gen, String name, ByteString value,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel, MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        return writeAndRenameAndDeleteAndConcat(
            writeClass,
            tabletId, gen,
            List.of(new KikimrKvClient.Write(name, value, storageChannel, priority)),
            List.of(), List.of(), List.of());
    }

    @CheckReturnValue
    public CompletableFuture<Void> writeAndRenameAndDelete(WriteClass writeClass, long tabletId, long gen, List<KikimrKvClient.Write> writes, List<KikimrKvClient.Rename> renames, List<NameRange> deletes) {
        return writeAndRenameAndDeleteAndConcat(
            writeClass,
            tabletId, gen,
            writes,
            renames,
            deletes,
            List.of());
    }

    @CheckReturnValue
    public CompletableFuture<Optional<byte[]>> readData(ReadClass readClass, long tabletId, long gen, String name) {
        var future = kikimrKvClient.readData(tabletId, gen, name, expiredAt(readClass), priority(readClass));
        measureAsyncReadOp(future, readClass, 1, value -> value.map(bytes -> bytes.length).orElse(0));
        return future;
    }

    @CheckReturnValue
    public CompletableFuture<byte[][]> readDataMulti(ReadClass readClass, long tabletId, long gen, KvChunkAddress[] addresses) {
        var future = kikimrKvClient.readDataMulti(tabletId, gen, addresses, expiredAt(readClass), priority(readClass));
        measureAsyncReadOp(future, readClass, addresses.length, value -> Arrays.stream(value)
            .filter(Objects::nonNull)
            .mapToLong(b -> b.length)
            .sum());

        return future;
    }

    @CheckReturnValue
    public CompletableFuture<byte[]> readDataLarge(ReadClass readClass, long tabletId, long gen, String key) {
        var future = kikimrKvClient.readDataLarge(tabletId, gen, key, expiredAt(readClass), priority(readClass));
        measureAsyncReadOp(future, readClass, 1, r -> r.length);
        return future;
    }

    @CheckReturnValue
    public CompletableFuture<byte[]> readDataLarge(ReadClass readClass, long tabletId, long gen, KvChunkAddress address) {
        var future = kikimrKvClient.readDataLarge(tabletId, gen, address.getOffset(), address.getLength(), address.getName(), expiredAt(readClass), priority(readClass));
        measureAsyncReadOp(future, readClass, 1, r -> r.length);
        return future;
    }

    @CheckReturnValue
    public CompletableFuture<byte[][]> readDataLargeMulti(ReadClass readClass, long tabletId, long gen, KvChunkAddress[] addresses) {
        var future = kikimrKvClient.readDataLargeMulti(tabletId, gen, addresses, expiredAt(readClass), priority(readClass));
        measureAsyncReadOp(future, readClass, addresses.length, r -> Arrays.stream(r)
            .mapToLong(b -> b.length)
            .sum());

        return future;
    }

    public CompletableFuture<Void> concatAndDeleteOriginals(WriteClass writeClass, long tabletId, long gen, List<String> names, String name) {
        return writeAndRenameAndDeleteAndConcat(writeClass, tabletId, gen, List.of(), List.of(), List.of(), List.of(new KikimrKvClient.Concat(names, name, false)));
    }

    public CompletableFuture<Long> incrementGeneration(long tabletId) {
        var future = kikimrKvClient.incrementGeneration(tabletId, expiredAt(WriteClass.OTHER));
        metrics.getWriteMetrics(WriteClass.OTHER).async.forFuture(future);
        return future;
    }

    public CompletableFuture<Void> deleteRange(WriteClass writeClass, long tabletId, long tabletGen, NameRange nameRange) {
        return writeAndRenameAndDeleteAndConcat(writeClass, tabletId, tabletGen, List.of(), List.of(), List.of(nameRange), List.of());
    }

    public CompletableFuture<Void> deleteRanges(WriteClass writeClass, long tabletId, long tabletGen, List<NameRange> nameRanges) {
        return writeAndRenameAndDeleteAndConcat(writeClass, tabletId, tabletGen, List.of(), List.of(), nameRanges, List.of());
    }

    public CompletableFuture<List<KikimrKvClient.KvEntryStats>> readRangeNames(
        ReadClass readClass, long tabletId, long tabletGen, NameRange nameRange)
    {
        var reader = new KvListFiles(kikimrKvClient, tabletId, tabletGen, clock, readClassTimeouts.get(readClass));
        var future = reader.listFiles(nameRange);
        measureAsyncReadOp(future, readClass, 1, value -> 0);
        return future;
    }

    private <T> void measureAsyncReadOp(CompletableFuture<T> future, ReadClass kind, long ops, ToLongFunction<T> read) {
        ReadMetrics readMetrics = metrics.getReadMetrics(kind);
        readMetrics.async.callStarted(ops);
        long startTimeNanos = System.nanoTime();
        future.whenComplete((r, e) -> {
                long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
                if (e != null) {
                    readMetrics.callCompletedError(elapsedMillis, ops, e);
                    return;
                }

                readMetrics.async.callCompletedOk(elapsedMillis, ops);
                readMetrics.inboundBytes.add(read.applyAsLong(r));
            });
    }

    private long expiredAt(ReadClass readClass) {
        long timeout = readClassTimeouts.get(readClass);
        if (timeout == 0) {
            return 0;
        }

        return clock.millis() + timeout;
    }

    private MsgbusKv.TKeyValueRequest.EPriority priority(ReadClass readClass) {
        switch (readClass) {
            case MERGE_READ_CHUNK:
                return MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND;
            default:
                return MsgbusKv.TKeyValueRequest.EPriority.REALTIME;
        }
    }

    private long expiredAt(WriteClass writeClass) {
        long timeout = writeClassTimeouts.get(writeClass);
        if (timeout == 0) {
            return 0;
        }

        return clock.millis() + timeout;
    }
}
