package ru.yandex.stockpile.kikimrKv.counting;

import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;

import io.grpc.Status;
import org.springframework.stereotype.Component;

import ru.yandex.kikimr.client.KikimrAnyResponseException;
import ru.yandex.kikimr.client.ResponseStatus;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Vladimir Gordiychuk
 */
@Component
public class KikimrKvClientMetrics {
    private final Map<ReadClass, ReadMetrics> readMetrics;
    private final Map<WriteClass, WriteMetrics> writeMetrics;

    public KikimrKvClientMetrics(MetricRegistry registry) {
        Map<ReadClass, ReadMetrics> readMetrics = new EnumMap<>(ReadClass.class);
        for (ReadClass readClass : ReadClass.values()) {
            readMetrics.put(readClass, new ReadMetrics(registry.subRegistry(Labels.of("class", readClass.name()))));
        }
        this.readMetrics = readMetrics;

        Map<WriteClass, WriteMetrics> writeMetrics = new EnumMap<>(WriteClass.class);
        for (WriteClass writeClass : WriteClass.values()) {
            writeMetrics.put(writeClass, new WriteMetrics(registry.subRegistry(Labels.of("class", writeClass.name()))));
        }
        this.writeMetrics = writeMetrics;
    }

    ReadMetrics getReadMetrics(ReadClass kind) {
        return readMetrics.get(kind);
    }

    WriteMetrics getWriteMetrics(WriteClass kind) {
        return writeMetrics.get(kind);
    }

    static class ReadMetrics {
        final MetricRegistry registry;
        final AsyncMetrics async;
        final Rate inboundBytes;
        final ConcurrentMap<String, Rate> errors = new ConcurrentHashMap<>();

        ReadMetrics(MetricRegistry registry) {
            this.registry = registry;
            async = new AsyncMetrics(registry, "kikimrKvClient.read.");
            inboundBytes = registry.rate("kikimrKvClient.read.inBoundBytes");
        }

        public void callCompletedError(long elapsedMillis, long count, @Nullable Throwable e) {
            async.callCompletedError(elapsedMillis, count);
            Rate rate = errors.computeIfAbsent(errorStatusName(e), status -> registry.rate("kikimrKvClient.read.errors", Labels.of("code", status)));
            rate.inc();
        }
    }

    static class WriteMetrics {
        final MetricRegistry registry;
        final AsyncMetrics async;
        final Rate outboundBytes;
        final ConcurrentMap<String, Rate> errors = new ConcurrentHashMap<>();
        final ConcurrentMap<EStorageChannel, Rate> opByChannel = new ConcurrentHashMap<>();

        WriteMetrics(MetricRegistry registry) {
            this.registry = registry;
            async = new AsyncMetrics(registry, "kikimrKvClient.write.");
            outboundBytes = registry.rate("kikimrKvClient.write.outBoundBytes");
        }

        public void callCompletedError(long elapsedMillis, long count, @Nullable Throwable e) {
            async.callCompletedError(elapsedMillis, count);
            Rate rate = errors.computeIfAbsent(errorStatusName(e), status -> registry.rate("kikimrKvClient.write.errors", Labels.of("code", status)));
            rate.inc();
        }

        public void callByChannel(EStorageChannel channel) {
            var rate = opByChannel.get(channel);
            if (rate == null) {
                rate = registry.rate("kikimrKvClient.write.byChannel", Labels.of("channel", channel.name()));
                opByChannel.put(channel, rate);
            }
            rate.inc();
        }
    }

    static String errorStatusName(@Nullable Throwable e) {
        if (e == null) {
            return ResponseStatus.MSTATUS_REJECTED.name();
        }

        Throwable cause = e;
        while (cause != null) {
            if (cause instanceof KikimrAnyResponseException) {
                KikimrAnyResponseException kve = (KikimrAnyResponseException) cause;
                MsgbusKv.TKeyValueResponse response = (MsgbusKv.TKeyValueResponse) kve.getResponse();
                ResponseStatus status = ResponseStatus.valueOf(response.getStatus());
                if (status != ResponseStatus.MSTATUS_UNKNOWN || response.getStatus() == 0) {
                    return status.name();
                } else {
                    return String.valueOf(response.getStatus());
                }
            }

            cause = cause.getCause();
        }

        return Status.fromThrowable(e).getCode().name();
    }
}
