package ru.yandex.stockpile.memState;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.log.StockpileLogEntryContent;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;

import static java.util.Objects.requireNonNull;
import static ru.yandex.stockpile.server.shard.MetricArchives.typeSafeAppend;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class LogEntriesContent implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(LogEntriesContent.class);

    private final MetricToArchiveMap metricToArchiveMap;
    private final Int2LongMap producerSeqNoById;
    private final DeletedShardSet deletedShards;
    private int fileCount = 0;
    private long fileSize = 0;

    public LogEntriesContent() {
        this.metricToArchiveMap = new MetricToArchiveMap();
        this.producerSeqNoById = new Int2LongOpenHashMap();
        this.deletedShards = new DeletedShardSet();
    }

    public LogEntriesContent(int metricsCapacity) {
        this.metricToArchiveMap = new MetricToArchiveMap(metricsCapacity);
        this.producerSeqNoById = new Int2LongOpenHashMap();
        this.deletedShards = new DeletedShardSet();
    }

    public LogEntriesContent(MetricToArchiveMap metricToArchiveMap, DeletedShardSet deletedShards, Int2LongMap producerSeqNoById, int fileCount, long fileSize) {
        this.metricToArchiveMap = requireNonNull(metricToArchiveMap);
        this.producerSeqNoById = producerSeqNoById;
        this.deletedShards = deletedShards;
        this.fileCount = fileCount;
        this.fileSize = fileSize;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
                + metricToArchiveMap.memorySizeIncludingSelf()
                + MemoryCounter.int2LongMapSize(producerSeqNoById)
                + deletedShards.memorySizeIncludingSelf();
    }

    public MetricToArchiveMap getMetricToArchiveMap() {
        return metricToArchiveMap;
    }

    public DeletedShardSet getDeletedShards() {
        return deletedShards;
    }

    public int getMetricCount() {
        return metricToArchiveMap.size();
    }

    public Int2LongMap getProducerSeqNoById() {
        return producerSeqNoById;
    }

    public long getProducerSeqNo(int producerId) {
        return producerSeqNoById.get(producerId);
    }

    public int getFileCount() {
        return fileCount;
    }

    public long getFileSize() {
        return fileSize;
    }

    public SizeAndCount sizeAndCount() {
        return new SizeAndCount(fileSize, fileCount);
    }

    public void updateWithLogEntry(int shardId, StockpileLogEntryContent logEntry) {
        AtomicLong diff = new AtomicLong();
        List<Metric> metricsNew = logEntry.getDataByMetricId()
                .long2ObjectEntrySet()
                .parallelStream()
                .map(entry -> {
                    MetricArchiveMutable target = metricToArchiveMap.getById(entry.getLongKey());
                    if (target == null) {
                        MetricArchiveMutable copy = MetricArchiveUtils.copy(StockpileFormat.CURRENT, entry.getValue());
                        return new Metric(entry.getLongKey(), copy);
                    }

                    long oldSize = target.memorySizeIncludingSelf();
                    typeSafeAppend(shardId, entry.getLongKey(), target, entry.getValue());
                    long newSize = target.memorySizeIncludingSelf();
                    diff.addAndGet(newSize - oldSize);
                    return null;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        metricToArchiveMap.addedBytesToArchive(diff.get());

        for (Metric e : metricsNew) {
            metricToArchiveMap.replace(e.localId, e.archive);
        }
        producerSeqNoById.putAll(logEntry.getProducerSeqNoById());
        deletedShards.addAll(logEntry.getDeletedShards());
    }

    public void addSize(long fileCount, long fileSize) {
        this.fileCount += fileCount;
        this.fileSize += fileSize;
    }

    public void updateWithLogEntriesContent(int shardId, LogEntriesContent sinceLastSnapshot) {
        updateWithLogEntry(shardId, sinceLastSnapshot.asLogEntryContent());
        addSize(sinceLastSnapshot.fileCount, sinceLastSnapshot.fileSize);
    }

    public boolean isEmpty() {
        return metricToArchiveMap.isEmpty() && deletedShards.isEmpty();
    }

    @Nonnull
    public StockpileLogEntryContent asLogEntryContent() {
        return new StockpileLogEntryContent(metricToArchiveMap.getMetrics(), deletedShards, producerSeqNoById);
    }

    public void release() {
        metricToArchiveMap.release();
    }

    private static class Metric {
        final long localId;
        final MetricArchiveMutable archive;

        public Metric(long localId, MetricArchiveMutable archive) {
            this.localId = localId;
            this.archive = archive;
        }
    }
}
