package ru.yandex.stockpile.memState;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

/**
 * @author Stepan Koltsov
 */
public class MetricToArchiveMap implements MemMeasurable, EstimateSerializedSize, Cloneable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(MetricToArchiveMap.class);

    private Long2ObjectOpenHashMap<MetricArchiveMutable> metrics;
    private long memorySizeInArchives;

    public MetricToArchiveMap() {
        this.metrics = new Long2ObjectOpenHashMap<>();
    }

    public MetricToArchiveMap(int capacity) {
        this.metrics = new Long2ObjectOpenHashMap<>(capacity);
    }

    public MetricToArchiveMap(Long2ObjectOpenHashMap<MetricArchiveMutable> metrics, long memorySizeInArchives) {
        this.metrics = metrics;
        this.memorySizeInArchives = memorySizeInArchives;
    }

    public Long2ObjectOpenHashMap<MetricArchiveMutable> getMetrics() {
        return metrics;
    }

    public boolean isEmpty() {
        return metrics.isEmpty();
    }

    private void updateMetric(long localId, MetricArchiveGeneric update) {
        try {
            MetricArchiveMutable metricArchive = metrics.get(localId);
            long oldSize = MemMeasurable.memorySizeOfNullable(metricArchive);
            if (metricArchive == null) {
                metricArchive = new MetricArchiveMutable();
                metrics.put(localId, metricArchive);
            }

            metricArchive.updateWith(update);
            long newSize = metricArchive.memorySizeIncludingSelf();
            memorySizeInArchives += newSize - oldSize;
        } catch (Exception e) {
            throw new RuntimeException("failed to update metric: " + StockpileLocalId.toString(localId), e);
        }
    }

    public void addRecordsMultiArchive(long localId, MetricArchiveGeneric update) {
        updateMetric(localId, update);
    }

    public void addedBytesToArchive(long delta) {
        memorySizeInArchives += delta;
    }

    @Nullable
    public MetricArchiveMutable getById(long localId) {
        return metrics.get(localId);
    }

    public MetricArchiveMutable getArchiveRef(long localId) {
        MetricArchiveMutable metricArchive = metrics.get(localId);
        if (metricArchive == null) {
            metricArchive = new MetricArchiveMutable();
            metrics.put(localId, metricArchive);
        }
        return metricArchive;
    }

    @Nonnull
    public MetricArchiveMutable getByIdOrEmpty(long localId) {
        return metrics.getOrDefault(localId, new MetricArchiveMutable());
    }

    public void update(Long2ObjectMap<MetricArchiveMutable> delta) {
        for (var entry : delta.long2ObjectEntrySet()) {
            addRecordsMultiArchive(entry.getLongKey(), entry.getValue());
        }
    }

    public void replace(long localId, MetricArchiveMutable archive) {
        MetricArchiveMutable prev = metrics.put(localId, archive);
        memorySizeInArchives += archive.memorySizeIncludingSelf();
        if (prev != null) {
            memorySizeInArchives -= prev.memorySizeIncludingSelf();
            prev.close();
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + memorySizeInArchives + MemoryCounter.long2ObjectOpenHashMapSize(metrics);
    }

    @Override
    public long estimateSerializedSize() {
        return memorySizeInArchives + metrics.size() * 5;
    }

    @Override
    public MetricToArchiveMap clone() {
        try {
            MetricToArchiveMap r = (MetricToArchiveMap) super.clone();
            r.metrics = deepCopy(r.metrics);

            for (Long2ObjectMap.Entry<MetricArchiveMutable> e : this.metrics.long2ObjectEntrySet()) {
                r.metrics.put(e.getLongKey(), e.getValue().clone());
            }

            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    private static Long2ObjectOpenHashMap<MetricArchiveMutable> deepCopy(Long2ObjectOpenHashMap<MetricArchiveMutable> map) {
        Long2ObjectOpenHashMap<MetricArchiveMutable> r = new Long2ObjectOpenHashMap<>();
        var it = map.long2ObjectEntrySet().fastIterator();
        while (it.hasNext()) {
            var entry = it.next();
            r.put(entry.getLongKey(), new MetricArchiveMutable(entry.getValue()));
        }
        return r;
    }

    public int size() {
        return metrics.size();
    }

    public void release() {
        for (var archive : metrics.values()) {
            archive.close();
        }
        metrics.clear();
    }
}
