package ru.yandex.stockpile.server;

import java.util.Arrays;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public enum SnapshotLevel {
    TWO_HOURS(
            "2h",
            File.of("s0.", FileKind.TWO_HOUR_CHUNK, WriteClass.SNAPSHOT_2H_CHUNK),
            File.of("i0.", FileKind.TWO_HOUR_INDEX, WriteClass.SNAPSHOT_2H_INDEX),
            File.of("c0.", FileKind.TWO_HOUR_COMMAND, WriteClass.SNAPSHOT_2H_COMMAND)
    ),
    DAILY(
            "dl",
            File.of("s1.", FileKind.DAILY_CHUNK, WriteClass.SNAPSHOT_DAILY_CHUNK),
            File.of("i1.", FileKind.DAILY_INDEX, WriteClass.SNAPSHOT_DAILY_INDEX),
            File.of("c1.", FileKind.DAILY_COMMAND, WriteClass.SNAPSHOT_DAILY_COMMAND)
    ),
    ETERNITY("et",
            File.of("s2.", FileKind.ETERNITY_CHUNK, WriteClass.SNAPSHOT_ETERNITY_CHUNK),
            File.of("i2.", FileKind.ETERNITY_INDEX, WriteClass.SNAPSHOT_ETERNITY_INDEX),
            File.of("c2.", FileKind.ETERNITY_COMMAND, WriteClass.SNAPSHOT_ETERNITY_COMMAND)
    ),
    ;

    private static final SnapshotLevel[] VALUES = values();

    // also used for label name
    public final String shortName;
    public final File chunk;
    public final File index;
    public final File command;

    SnapshotLevel(String shortName, File chunk, File index, File command)
    {
        this.shortName = shortName;

        this.chunk = chunk;
        this.index = index;
        this.command = command;
    }

    public static SnapshotLevel[] inOrderOfRead() {
        return new SnapshotLevel[]{ETERNITY, DAILY, TWO_HOURS};
    }

    private static ParserFormatter<SnapshotLevel> prefixParserFormatter(Function<SnapshotLevel, String> f) {
        return new ParserFormatter<SnapshotLevel>() {
            @Override
            public String format(SnapshotLevel snapshotLevel) {
                return f.apply(snapshotLevel);
            }

            @Override
            public Result<SnapshotLevel> parsePrefix(String string) {
                for (SnapshotLevel snapshotLevel : VALUES) {
                    String prefix = f.apply(snapshotLevel);
                    if (string.startsWith(prefix)) {
                        return new Success<>(snapshotLevel, string.substring(prefix.length()));
                    }
                }

                return new Failure<>();
            }

            @Override
            public String toString() {
                return "snapshot" + Stream.of(VALUES).map(f).collect(Collectors.joining("|", "(", ")"));
            }
        };
    }

    public static ParserFormatter<SnapshotLevel> chunkPrefix0ParserFormatter() {
        return prefixParserFormatter(l -> l.chunk.prefix);
    }

    public static ParserFormatter<SnapshotLevel> indexPrefix0ParserFormatter() {
        return prefixParserFormatter(l -> l.index.prefix);
    }

    public static ParserFormatter<SnapshotLevel> commandPrefixParserFormatter() {
        return prefixParserFormatter(l -> l.command.prefix);
    }

    public SnapshotLevel[] currentAndAboveLevels() {
        return Arrays.copyOfRange(VALUES, 0, ordinal() + 1);
    }

    public SnapshotLevel nextLevel() {
        return VALUES[this.ordinal() + 1];
    }

    public static class File {
        private final String prefix;
        private final String currentPrefix;
        private final FileKind kind;
        private final WriteClass writeClass;

        public File(String prefix, String currentPrefix, FileKind kind, WriteClass writeClass) {
            this.prefix = prefix;
            this.currentPrefix = currentPrefix;
            this.kind = kind;
            this.writeClass = writeClass;
        }

        public String prefix() {
            return prefix;
        }

        public String currentPrefix() {
            return currentPrefix;
        }

        public FileKind kind() {
            return kind;
        }

        public WriteClass writeClass() {
            return writeClass;
        }

        public static File of(String prefix, FileKind kind, WriteClass writeClass) {
            return new File(prefix, StockpileKvNames.CURRENT_PREFIX + prefix, kind, writeClass);
        }
    }
}
