package ru.yandex.stockpile.server;

import java.util.Optional;

import javax.annotation.Nullable;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.kikimr.client.kv.KikimrClientProxy;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientImpl;
import ru.yandex.kikimr.client.kv.KikimrKvV2ClientImpl;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.kikimr.KikimrTransportFactory;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.ydb.YdbAuthProviders;
import ru.yandex.solomon.ydb.YdbClients;


/**
 * @author Stepan Koltsov
 */
@Configuration
public class StockpileKikimrClientContext implements DisposableBean {

    private final TKikimrClientConfig config;
    private final KikimrTransportFactory transportFactory;
    private final YdbClients ydbClients;
    private final YdbAuthProviders authProviders;

    @Autowired
    public StockpileKikimrClientContext(
            @Qualifier("KikimrClientConfig") TKikimrClientConfig config,
            ThreadPoolProvider threadPoolProvider,
            MetricRegistry registry,
            Optional<IamTokenClient> iamTokenClient,
            SecretProvider secrets)
    {
        this.config = config;
        this.transportFactory = new KikimrTransportFactory("KikimrClientConfig", threadPoolProvider, registry);
        this.authProviders = new YdbAuthProviders(iamTokenClient.orElse(null), threadPoolProvider, secrets);
        this.ydbClients = new YdbClients("KikimrClientConfig", config, threadPoolProvider, registry, authProviders);
    }


    @Bean
    @Primary
    KikimrKvClient kikimrKvClientProxy(@Qualifier("kikimrClient") KikimrKvClient clientV1, @Qualifier("kikimrKvClientV2") @Nullable KikimrKvClient clientV2) {
        boolean isV2Enabled = config.hasConnectionConfig() && config.getConnectionConfig().hasIsV2Client() && config.getConnectionConfig().getIsV2Client();
        return new KikimrClientProxy(clientV1, clientV2, isV2Enabled);
    }

    @Bean("kikimrKvClientV2")
    KikimrKvClient kikimrKvClientV2() {
        if (!config.hasConnectionConfig() || !config.getConnectionConfig().hasVolumePath()) {
            return null;
        }
        String volumePath = config.getConnectionConfig().getVolumePath();
        return new KikimrKvV2ClientImpl(transportFactory.newNodeRouter(config, authProviders), volumePath);
    }

    @Bean("kikimrClient")
    KikimrKvClient kikimrClient() {
        return new KikimrKvClientImpl(transportFactory.newTransport(config));
    }

    @Bean
    SchemeClient schemeClient() {
        return ydbClients.getSchemeClient();
    }

    @Bean
    TableClient tableClient() {
        return ydbClients.getTableClient();
    }

    @Bean
    MetricSupplier tableClientMetrics() {
        return ydbClients.getTableClientMetrics();
    }

    @Override
    public void destroy() {
        ydbClients.close();
    }
}
