package ru.yandex.stockpile.server;

import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.OptionalSet;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.stockpile.TKvChannelConfig;
import ru.yandex.solomon.config.protobuf.stockpile.TKvChannelConfig.Channels;
import ru.yandex.solomon.config.protobuf.stockpile.TKvStorageConfig;
import ru.yandex.solomon.config.protobuf.stockpile.TReadConfig;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;
import ru.yandex.stockpile.kikimrKv.KvChannel;
import ru.yandex.stockpile.kikimrKv.KvChannels;
import ru.yandex.stockpile.kikimrKv.KvFiles;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.data.dao.ReadBatcher;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.shard.StockpileExecutor;
import ru.yandex.stockpile.server.shard.StockpileReadExecutor;
import ru.yandex.stockpile.server.shard.StockpileScheduledExecutor;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class StockpileKvPersistenceContext {
    private final TKvStorageConfig config;
    private final MetricRegistry registry;

    public StockpileKvPersistenceContext(MetricRegistry registry, TKvStorageConfig config) {
        this.registry = registry;
        this.config = config;
    }

    @Bean
    public KikimrKvClientCounting kikimrKvClientCounting(KikimrKvClient client, Clock clock) {
        KikimrKvClientMetrics metrics = new KikimrKvClientMetrics(registry);
        long defaultTimeout = TimeConverter.protoToDuration(config.getDefaultRequestTimeout()).toMillis();

        EnumMapToLong<ReadClass> readTimeouts = new EnumMapToLong<>(ReadClass.class, defaultTimeout);
        EnumMapToLong<WriteClass> writeTimeouts = new EnumMapToLong<>(WriteClass.class, defaultTimeout);

        OptionalSet.setTimeMillis(millis -> {
            readTimeouts.set(ReadClass.START_READ_LOG, millis);
            readTimeouts.set(ReadClass.START_READ_INDEX, millis);
        }, config.getLogsReadTimeout());

        OptionalSet.setTimeMillis(millis -> {
            writeTimeouts.set(WriteClass.LOG, millis);
            writeTimeouts.set(WriteClass.LOG_SNAPSHOT, millis);
        }, config.getLogsWriteTimeout());

        OptionalSet.setTimeMillis(millis -> {
            readTimeouts.set(ReadClass.MERGE_READ_CHUNK, millis);
        }, config.getSnapshotReadTimeout());

        OptionalSet.setTimeMillis(millis -> {
            writeTimeouts.set(WriteClass.SNAPSHOT_2H_CHUNK, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_2H_INDEX, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_2H_COMMAND, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_DAILY_CHUNK, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_DAILY_INDEX, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_DAILY_COMMAND, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_ETERNITY_CHUNK, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_ETERNITY_INDEX, millis);
            writeTimeouts.set(WriteClass.SNAPSHOT_ETERNITY_COMMAND, millis);
        }, config.getSnapshotWriteTimeout());

        return new KikimrKvClientCounting(client, metrics, clock, readTimeouts, writeTimeouts);
    }

    @Bean
    public ReadBatcher readBatcher(KikimrKvClientCounting kvClient, @StockpileReadExecutor ExecutorService executor, Optional<TReadConfig> config) {
        int inflightLimit = config.map(TReadConfig::getDiskReadInFlightLimit).filter(v -> v > 0).orElse(500);
        int inflightLimitPerShard = config.map(TReadConfig::getDiskReadInFlightLimitPerShard).filter(v -> v > 0).orElse(10);
        return new ReadBatcher(executor, kvClient, inflightLimit, inflightLimitPerShard, registry);
    }

    @Bean
    @Lazy
    public KvTabletsMapping kvTabletsMapping(
        KikimrKvClient kvClient,
        @StockpileExecutor ExecutorService executor,
        @StockpileScheduledExecutor ScheduledExecutorService timer)
    {
        return new KvTabletsMapping(config.getVolumePath(), kvClient, executor, timer);
    }

    @Bean
    KvChannels channelsConfig(@Nullable TKvChannelConfig config) {
        var channels = new KvChannels();
        if (config == null) {
            return channels;
        }

        setChannel(channels, FileKind.LOG, config.getLog());
        setChannel(channels, FileKind.LOG_SNAPSHOT, config.getLogSnapshot());
        setChannel(channels, FileKind.ETERNITY_CHUNK, config.getEternityChunk());
        setChannel(channels, FileKind.ETERNITY_INDEX, config.getEternityIndex());
        setChannel(channels, FileKind.ETERNITY_COMMAND, config.getEternityCommand());
        setChannel(channels, FileKind.DAILY_CHUNK, config.getDailyChunk());
        setChannel(channels, FileKind.DAILY_INDEX, config.getDailyIndex());
        setChannel(channels, FileKind.DAILY_COMMAND, config.getDailyCommand());
        setChannel(channels, FileKind.TWO_HOUR_CHUNK, config.getTwoHourChunk());
        setChannel(channels, FileKind.TWO_HOUR_INDEX, config.getTwoHourIndex());
        setChannel(channels, FileKind.TWO_HOUR_COMMAND, config.getTwoHourCommand());
        setChannel(channels, FileKind.PRODUCER_SEQUENCES, config.getProducerSequence());

        var fallback = toKvChannel(config.getInlineFallback());
        if (fallback != null) {
            channels.changeInlineFallback(fallback);
        }

        return channels;
    }

    @Bean
    KvFiles kvFiles(KikimrKvClient kvClient) {
        return new KvFiles(kvClient, config.getVolumePath());
    }

    void setChannel(KvChannels channels, FileKind fileKind, Channels channel) {
        var kvChannel = toKvChannel(channel);
        if (kvChannel == null) {
            return;
        }

        channels.changeChannel(fileKind, kvChannel);
    }

    @Nullable
    private KvChannel toKvChannel(Channels channel) {
        switch (channel) {
            case INLINE:
                return KvChannel.INLINE;
            case HDD:
                return KvChannel.HDD;
            case SSD1:
                return KvChannel.SSD1;
            case SSD2:
                return KvChannel.SSD2;
            case SSD3:
                return KvChannel.SSD3;
            case SSD4:
                return KvChannel.SSD4;
            default:
                return null;
        }
    }

}
