package ru.yandex.stockpile.server.cache;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.staffOnly.manager.table.TableColumn;
import ru.yandex.stockpile.server.cache.lhm.Long2ObjectLinkedHashMap;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
@NotThreadSafe
public class Long2ObjectLruCache<V extends MemMeasurable & AutoCloseable> implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Long2ObjectLruCache.class);

    private Long2ObjectLinkedHashMap<V> map = new Long2ObjectLinkedHashMap<>();

    private long maxSizeBytes = 0;

    private long addedRecords = 0;
    private long removedRecords = 0;

    private long addedBytes = 0;
    private long removedBytes = 0;

    private void addedRecordUpdateCounters(long sizeDelta) {
        addedRecords += 1;
        addedBytes += sizeDelta;
    }

    private void removeRecordUpdateCounters(long sizeDelta) {
        removedRecords += 1;
        removedBytes += sizeDelta;
    }

    // can be used from another thread
    public void setMaxSizeBytes(long maxSizeBytes) {
        if (maxSizeBytes < 0) {
            throw new IllegalArgumentException();
        }
        this.maxSizeBytes = maxSizeBytes;
    }

    public long getRecordsUsed() {
        return addedRecords - removedRecords;
    }

    public long getMaxSizeBytes() {
        return maxSizeBytes;
    }

    public long getAddedRecords() {
        return addedRecords;
    }

    public long getRemovedRecords() {
        return removedRecords;
    }

    public long getAddedBytes() {
        return addedBytes;
    }

    public long getRemovedBytes() {
        return removedBytes;
    }

    public long getBytesUsed() {
        return addedBytes - removedBytes;
    }

    @TableColumn
    public int size() {
        return map.size();
    }

    public boolean isEmpty() {
        return map.isEmpty();
    }

    public void clear() {
        V entry;
        while ((entry = map.removeOldest()) != null) {
            close(entry);
        }
        removedBytes = addedBytes;
        removedRecords = addedRecords;
    }

    public void removeEntriesAboveSizeLimit() {
        while (memorySizeIncludingSelf() >= maxSizeBytes) {
            V entry = map.removeOldest();
            if (entry == null) {
                return;
            }

            removeRecordUpdateCounters(entry.memorySizeIncludingSelf());
            close(entry);
        }
    }

    public void remove(long key) {
        V removed = map.remove(key);
        if (removed != null) {
            removeRecordUpdateCounters(removed.memorySizeIncludingSelf());
            close(removed);
        }
    }

    @Nullable
    public V get(long key) {
        return map.get(key);
    }

    public void replace(long key, V value) {
        V removed = map.put(key, value);

        if (removed == null) {
            addedRecordUpdateCounters(value.memorySizeIncludingSelf());
        } else {
            addedBytes += value.memorySizeIncludingSelf() - removed.memorySizeIncludingSelf();
            close(removed);
        }
    }

    public void sizeUpdated(long diff) {
        addedBytes += diff;
    }

    @TableColumn
    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
            + map.memorySizeIncludingSelf()
            + (addedBytes - removedBytes);
    }

    private void close(AutoCloseable closable) {
        try {
            closable.close();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() {
        clear();
    }
}
