package ru.yandex.stockpile.server.cache.intrList;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.misc.lang.Validate;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
@NotThreadSafe
public class IntrusiveList<A extends IntrusiveListItem<A>> implements Iterable<A>, MemMeasurable {
    public static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IntrusiveList.class);

    @Nullable
    private A head;
    private int size = 0;

    public void checkLinksForTest() {
        if (head == null) {
            Validate.equals(0, size);
            return;
        }

        int size = 0;
        A a = head;
        do {
            Validate.isTrue(a.next.prev == a);
            Validate.isTrue(a.prev.next == a);
            a = a.next;
            ++size;
        } while (a != head);
        Validate.equals(size, this.size);
    }

    public void add(A a) {
        if (a.isInList()) {
            throw new IllegalArgumentException();
        }

        if (head == null) {
            a.prev = a.next = a;
            head = a;
        } else {
            head.prev.next = a;
            a.prev = head.prev;
            head.prev = a;
            a.next = head;
        }
        ++size;
    }

    public void add(int index, A a) {
        if (a.isInList()) {
            throw new IllegalArgumentException();
        }

        if (index > size) {
            throw new IllegalArgumentException();
        }

        if (index == size) {
            add(a);
        } else {
            A loc = get(index);
            a.next = loc;
            a.prev = loc.prev;
            loc.prev.next = a;
            loc.prev = a;
            if (index == 0) {
                head = a;
            }
            ++size;
        }
    }

    @Nonnull
    public A removeFirst() {
        if (head == null) {
            throw new IllegalStateException();
        }
        A r = head;
        remove(head);
        return r;
    }

    public void moveToBack(A a) {
        if (!a.isInList()) {
            throw new IllegalArgumentException();
        }

        if (size == 1) {
            return;
        }

        remove(a);
        add(a);
    }

    public void remove(A a) {
        if (!a.isInList()) {
            throw new IllegalStateException();
        }

        if (a.next == a) {
            head = null;
        } else {
            if (a == head) {
                head = a.next;
            }
            a.next.prev = a.prev;
            a.prev.next = a.next;
        }

        a.prev = a.next = null;
        --size;
    }

    public void removeAt(int index) {
        remove(get(index));
    }

    @Nonnull
    public List<A> toList() {
        List<A> r = new ArrayList<>();
        forEach(r::add);
        return r;
    }

    @Nonnull
    public A get(int index) {
        return iterator().drop(index).next();
    }

    @Override
    public IteratorF<A> iterator() {
        return new IteratorF<A>() {
            private A next = head;

            @Override
            public boolean hasNext() {
                return next != null;
            }

            @Override
            public A next() {
                if (!hasNext()) {
                    throw new IllegalStateException();
                }
                A r = next;
                next = next.next;
                if (next == head) {
                    next = null;
                }
                return r;
            }
        };
    }

    public int size() {
        return size;
    }

    public boolean isEmpty() {
        return head == null;
    }

    @Override
    public String toString() {
        return iterator().mkString("[", ", ", "]");
    }

    /**
     * Does not detach entries for speed and simplicity.
     */
    public void clear() {
        head = null;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE;
    }
}
