package ru.yandex.stockpile.server.cache.lhm;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.misc.lang.Validate;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.cache.intrList.IntrusiveList;
import ru.yandex.stockpile.server.cache.intrList.IntrusiveListItem;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class Long2ObjectLinkedHashMap<V> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Long2ObjectLinkedHashMap.class);

    private Long2ObjectMap<V> map = new Long2ObjectMap<>();
    private IntrusiveList<EntryImpl<V>> entries = new IntrusiveList<>();

    public void clear() {
        map.clear();
        entries.clear();
    }

    public int size() {
        return entries.size();
    }

    public boolean isEmpty() {
        return entries.isEmpty();
    }

    @Nullable
    public V get(long key) {
        EntryImpl<V> entry = map.get(key);
        return entry != null ? entry.value : null;
    }

    @Nullable
    public V getAndRefresh(long key) {
        EntryImpl<V> entry = map.get(key);
        if (entry != null) {
            entries.moveToBack(entry);
            return entry.value;
        } else {
            return null;
        }
    }

    @Nonnull
    public V getAndRefreshOrCreate(long key, Supplier<V> create) {
        EntryImpl<V> entry = map.get(key);
        if (entry != null) {
            entries.moveToBack(entry);
        } else {
            entry = new EntryImpl<>(key, create.get());
            map.put(key, entry);
            entries.add(entry);
        }
        return entry.value;
    }

    @Nullable
    public V put(long key, V value) {
        EntryImpl<V> removed = map.remove(key);
        V r;
        if (removed != null) {
            entries.remove(removed);
            r = removed.value;
        } else {
            r = null;
        }

        EntryImpl<V> entry = new EntryImpl<>(key, value);
        map.put(key, entry);
        entries.add(entry);

        return r;
    }

    @Nullable
    public V removeOldest() {
        if (isEmpty()) {
            return null;
        }
        EntryImpl<V> entry = entries.removeFirst();
        EntryImpl<V> removed = map.remove(entry.key);
        if (removed != entry) {
            throw new IllegalStateException();
        }
        return entry.value;
    }

    @Nullable
    public V remove(long key) {
        EntryImpl<V> entry = map.remove(key);
        if (entry != null) {
            entries.remove(entry);
            return entry.value;
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        Iterator<EntryImpl<V>> it = entries.iterator();
        LinkedHashMap<Long, V> result = new LinkedHashMap<>();
        while (it.hasNext()) {
            EntryImpl<V> entry = it.next();
            result.put(entry.key, entry.value);
        }
        return result.toString();
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
            + map.memorySizeIncludingSelf()
            + IntrusiveList.SELF_SIZE
            + (EntryImpl.SELF_SIZE * entries.size());
    }

    public void checkForTest() {
        entries.checkLinksForTest();
        Validate.equals(map.size(), entries.size());
        for (EntryImpl<V> entry : entries) {
            Validate.isTrue(entry == map.get(entry.key));
        }
    }

    private static class Long2ObjectMap<V> extends Long2ObjectOpenHashMap<EntryImpl<V>> implements MemMeasurable {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Long2ObjectMap.class);

        @Override
        public long memorySizeIncludingSelf() {
            return SELF_SIZE
                + MemoryCounter.arrayObjectSize(key)
                + MemoryCounter.arrayObjectSize(value);
        }
    }

    private static class EntryImpl<V> extends IntrusiveListItem<EntryImpl<V>> {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(EntryImpl.class);

        @Nonnull
        private final long key;
        @Nonnull
        private final V value;

        public EntryImpl(long key, V value) {
            this.key = key;
            this.value = value;
        }

        @Override
        public String toString() {
            return key + "=" + value;
        }

        @Override
        public long memorySizeIncludingSelf() {
            return SELF_SIZE;
        }
    }
}
