package ru.yandex.stockpile.server.data;

import java.util.Iterator;

import it.unimi.dsi.fastutil.longs.LongIterator;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;
import org.jetbrains.annotations.NotNull;

import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.data.DeletedShardSet.Shard;

/**
 * @author Vladimir Gordiychuk
 */
public class DeletedShardSet implements MemMeasurable, Iterable<Shard> {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(DeletedShardSet.class);
    private final LongSet set;

    public DeletedShardSet() {
        this(0);
    }

    public DeletedShardSet(int capacity) {
        this.set = new LongOpenHashSet(capacity);
    }

    public void add(int projectId, int shardId) {
        set.add(key(projectId, shardId));
    }

    public void addAll(DeletedShardSet other) {
        set.addAll(other.set);
    }

    public boolean contains(int projectId, int shardId) {
        return set.contains(key(projectId, shardId));
    }

    public boolean isEmpty() {
        return set.isEmpty();
    }

    public int size() {
        return set.size();
    }

    static long key(int projectId, int shardId) {
        long result = (long) projectId << 32;
        result |= Integer.toUnsignedLong(shardId);
        return result;
    }

    static int projectId(long key) {
        return (int) (key >> 32);
    }

    static int shardId(long key) {
        return (int) (key & 0xffffffffL);
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.longOpenHashSetSize(set);
    }

    @Override
    public String toString() {
        return "DeletedShardSet{" +
                "size=" + set.size() +
                '}';
    }

    @NotNull
    @Override
    public Iterator<Shard> iterator() {
        return new Iter(set.iterator());
    }

    public static class Shard {
        public int projectId;
        public int shardId;
    }

    private static class Iter implements Iterator<Shard> {
        private final LongIterator it;
        private final Shard next = new Shard();

        public Iter(LongIterator it) {
            this.it = it;
        }

        @Override
        public boolean hasNext() {
            return it.hasNext();
        }

        @Override
        public Shard next() {
            long key = it.next();
            next.projectId = projectId(key);
            next.shardId = shardId(key);
            return next;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof DeletedShardSet)) return false;

        DeletedShardSet that = (DeletedShardSet) o;
        return set.equals(that.set);
    }

    @Override
    public int hashCode() {
        return set.hashCode();
    }
}
