package ru.yandex.stockpile.server.data;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.WithVersionHeaderSerializer;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;

/**
 * @author Stepan Koltsov
 *
 * @see ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer
 */
@ParametersAreNonnullByDefault
public class DeletedShardSetSerializer extends WithVersionHeaderSerializer<DeletedShardSet> {
    DeletedShardSetSerializer(StockpileFormat format) {
        super(DeletedShardSetSerializer::serializerForFormat, e -> format);
    }

    @Nonnull
    public static NakedSerializer<DeletedShardSet> serializerForFormat(StockpileFormat format) {
        if (format.lt(StockpileFormat.DELETED_SHARDS_39)) {
            return new V0();
        }

        return new V1();
    }

    private static class V0 implements NakedSerializer<DeletedShardSet> {

        @Override
        public DeletedShardSet deserializeToEofImpl(StockpileDeserializer deserializer) {
            return new DeletedShardSet();
        }

        @Override
        public void serializeToEof(DeletedShardSet deletedShardSet, StockpileSerializer serializer) {
            // noop
        }
    }

    private static class V1 implements NakedSerializer<DeletedShardSet> {
        @Override
        public void serializeToEof(DeletedShardSet deletedShards, StockpileSerializer serializer) {
            serializer.writeVarint32(deletedShards.size());
            for (var shard : deletedShards) {
                serializer.writeVarint32(shard.projectId);
                serializer.writeFixed32(shard.shardId);
            }
        }

        @Override
        public DeletedShardSet deserializeToEofImpl(StockpileDeserializer deserializer) {
            int size = deserializer.readVarint32();
            var result = new DeletedShardSet(size);
            for (int index = 0; index < size; index++) {
                int projectId = deserializer.readVarint32();
                int shardId = deserializer.readFixed32();
                result.add(projectId, shardId);
            }
            return result;
        }
    }
}
