package ru.yandex.stockpile.server.data.chunk;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.collection.array.UnsignedLongArrays;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ChunkIndex implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ChunkIndex.class);

    @Nonnull
    private long[] localIdsSorted;
    @Nonnull
    private long[] lastTssMillis;
    @Nonnull
    private int[] nextOffsets; // element <i> stores offset for next item <i+1>
    private int metricCount = 0;

    public ChunkIndex() {
        this.localIdsSorted = Cf.LongArray.emptyArray();
        this.lastTssMillis = Cf.LongArray.emptyArray();
        this.nextOffsets = Cf.IntegerArray.emptyArray();
    }

    public ChunkIndex(long[] localIdsSorted, long[] lastTssMillis, int[] nextOffsets) {
        this.localIdsSorted = localIdsSorted;
        this.lastTssMillis = lastTssMillis;
        this.nextOffsets = nextOffsets;
        this.metricCount = localIdsSorted.length;
    }

    public ChunkIndex(int capacity) {
        this.localIdsSorted = new long[capacity];
        this.lastTssMillis = new long[capacity];
        this.nextOffsets = new int[capacity];
    }

    public int metricCount() {
        return metricCount;
    }

    @Nonnull
    public long[] getLocalIdsSortedArray() {
        return localIdsSorted;
    }

    public long getLocalId(int index) {
        return localIdsSorted[index];
    }

    @Nonnull
    public long[] getLastTssMillisArray() {
        return lastTssMillis;
    }

    public long getLastTssMillis(int index) {
        return lastTssMillis[index];
    }

    @Nonnull
    public int[] getNextOffsetsArray() {
        return nextOffsets;
    }

    @Nonnull
    public int[] getSizesArray() {
        int[] sizes = new int[metricCount];
        int prevOffset = 0;
        for (int i = 0; i < metricCount; i++) {
            sizes[i] = nextOffsets[i] - prevOffset;
            prevOffset = nextOffsets[i];
        }
        return sizes;
    }

    public int getSize(int index) {
        int prevOffset = index > 0 ? nextOffsets[index - 1] : 0;
        return nextOffsets[index] - prevOffset;
    }

    public int getOffset(int index) {
        if (index == 0) {
            return 0;
        }

        return nextOffsets[index - 1];
    }

    public long countChunkDiskSize() {
        if (metricCount == 0) {
            return 0;
        }
        return nextOffsets[metricCount - 1];
    }

    public void addMetric(long localId, long lastTsMillis, int metricDataSize) {
        if (metricCount > 0 && StockpileLocalId.compare(localId, localIdsSorted[metricCount - 1]) <= 0) {
            throw new IllegalArgumentException("not sorted");
        }

        ArrayListImpl.prepareAdd(new ArrayListImpl.Accessor() {
            @Override
            public int capacity() {
                return localIdsSorted.length;
            }

            @Override
            public int size() {
                return metricCount;
            }

            @Override
            public void reserveExact(int newCapacity) {
                localIdsSorted = Arrays.copyOf(localIdsSorted, newCapacity);
                lastTssMillis = Arrays.copyOf(lastTssMillis, newCapacity);
                nextOffsets = Arrays.copyOf(nextOffsets, newCapacity);
            }
        });
        localIdsSorted[metricCount] = localId;
        lastTssMillis[metricCount] = lastTsMillis;
        if (metricCount == 0) {
            nextOffsets[metricCount] = metricDataSize;
        } else {
            nextOffsets[metricCount] = nextOffsets[metricCount - 1] + metricDataSize;
        }
        ++metricCount;
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE +
            MemoryCounter.arrayObjectSize(localIdsSorted) +
            MemoryCounter.arrayObjectSize(nextOffsets) +
            MemoryCounter.arrayObjectSize(lastTssMillis);
    }

    @Nullable
    public DataRange findMetricData(long localId) {
        if (metricCount == 0) {
            return null;
        }

        if (StockpileLocalId.compare(localId, localIdsSorted[0]) < 0
            || StockpileLocalId.compare(localId, localIdsSorted[metricCount - 1]) > 0)
        {
            return null;
        }

        final int index = UnsignedLongArrays.binarySearch(localIdsSorted, 0, metricCount, localId);
        if (index == 0) {
            return new DataRange(0, nextOffsets[0]);
        } else if (index > 0) {
            final int length = nextOffsets[index] - nextOffsets[index - 1];
            return new DataRange(nextOffsets[index - 1], length);
        }

        return null;
    }

    public long findMetricLastTsMillis(long localId) {
        if (metricCount == 0) {
            return -1;
        }

        if (StockpileLocalId.compare(localId, localIdsSorted[0]) < 0
            || StockpileLocalId.compare(localId, localIdsSorted[metricCount - 1]) > 0)
        {
            return -1;
        }

        final int index = UnsignedLongArrays.binarySearch(localIdsSorted, 0, metricCount, localId);
        if (index >= 0) {
            return lastTssMillis[index];
        }
        return -1;
    }

    @Override
    public String toString() {
        return "ChunkIndex{metricCount=" + metricCount + ",...}";
    }
}
