package ru.yandex.stockpile.server.data.chunk;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.HeapStockpileSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.dao.StockpileKvLimits;
import ru.yandex.stockpile.server.data.index.ChunkIndexArray;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.index.SnapshotIndexProperties;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsProject;
import ru.yandex.stockpile.server.shard.SnapshotReason;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ChunkWriter implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ChunkWriter.class);

    private final StockpileFormat indexFormat = StockpileFormat.CURRENT;
    private final long tsMillis;
    private final long decimatedAt;
    @Nonnull
    private final SnapshotReason snapshotReason;
    private final NakedSerializer<MetricArchiveImmutable> immutableDataSerializer;

    private long lastMetricId = StockpileLocalId.MIN;

    private final HeapStockpileSerializer chunkSerializer = new HeapStockpileSerializer(StockpileKvLimits.RECOMMENDED_MAX_FILE_SIZE);
    private ChunkIndexArray chunkIndexArray = new ChunkIndexArray();

    /// statistics

    // Total record count in all chunks
    private long recordCount = 0;
    // record count per project
    private IndexStatsProject stats = new IndexStatsProject();
    private long metricCount = 0;

    public ChunkWriter(SnapshotReason snapshotReason, long now, long decimatedAt) {
        this.tsMillis = now;
        this.decimatedAt = decimatedAt;
        this.snapshotReason = snapshotReason;
        this.immutableDataSerializer = SnapshotIndexContentSerializer.dataSerializerForVersionSealed(indexFormat);
    }

    @Nullable
    public ChunkWithNo writeMetric(long metricId, long lastTsMillis, MetricArchiveImmutable archive) {
        if (StockpileLocalId.compare(metricId, lastMetricId) <= 0) {
            throw new IllegalArgumentException("Last " + StockpileLocalId.toString(lastMetricId) + " >= " + StockpileLocalId.toString(metricId));
        }

        int before = chunkSerializer.size();
        immutableDataSerializer.serializeToEof(archive, chunkSerializer);
        int after = chunkSerializer.size();
        chunkIndexArray.addMetric(metricId, lastTsMillis, after - before);
        lastMetricId = metricId;
        updateStatistics(archive);
        return ensureChunkFull();
    }

    private void updateStatistics(MetricArchiveGeneric archive) {
        recordCount += archive.getRecordCount();
        metricCount += 1;
        stats.add(archive);
    }

    private ChunkWithNo ensureChunkFull() {
        if (chunkSerializer.size() >= StockpileKvLimits.RECOMMENDED_MAX_FILE_SIZE) {
            return closeChunk();
        } else {
            return null;
        }
    }

    @Nonnull
    private ChunkWithNo closeChunk() {
        ChunkWithNo r = new ChunkWithNo(chunkIndexArray.getChunksCount(), chunkSerializer.flush());
        chunkIndexArray.finishChunk();
        return r;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += chunkIndexArray.memorySizeIncludingSelf();
        size += chunkSerializer.memorySizeIncludingSelf();
        return size;
    }

    public static class Finish {
        public final SnapshotIndexContent index;
        @Nullable
        public final ChunkWithNo chunkWithNo;

        public Finish(SnapshotIndexContent index, @Nullable ChunkWithNo chunkWithNo) {
            this.index = index;
            this.chunkWithNo = chunkWithNo;
        }
    }

    /**
     * Returns index.
     */
    @Nonnull
    public Finish finish() {
        ChunkWithNo chunkWithNo;
        if (chunkSerializer.size() > 0) {
            chunkWithNo = closeChunk();
        } else {
            chunkWithNo = null;
        }

        chunkIndexArray.shrinkToFit();
        long createDurationMillis = System.currentTimeMillis() - tsMillis;
        SnapshotIndexProperties properties = new SnapshotIndexProperties()
            .setCreatedAt(tsMillis)
            .setCreateDurationMillis(createDurationMillis)
            .setDecimatedAt(decimatedAt)
            .setRecordCount(recordCount)
            .setSnapshotReason(snapshotReason)
            .setStatsByProject(stats)
            .setMetricCount(metricCount);
        SnapshotIndexContent index = new SnapshotIndexContent(
            indexFormat,
            properties,
            chunkIndexArray);
        return new Finish(index, chunkWithNo);
    }

}
