package ru.yandex.stockpile.server.data.command;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.WithVersionHeaderSerializer;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.DeletedShardSetSerializer;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SnapshotCommandContentSerializer extends WithVersionHeaderSerializer<SnapshotCommandContent> {
    public static final NakedSerializer<SnapshotCommandContent> S =
        new SnapshotCommandContentSerializer(StockpileFormat.CURRENT);

    SnapshotCommandContentSerializer(StockpileFormat format) {
        super(SnapshotCommandContentSerializer::serializerForFormat, e -> format);
    }

    @Nonnull
    public static NakedSerializer<SnapshotCommandContent> serializerForFormat(StockpileFormat format) {
        if (format.le(StockpileFormat.HISTOGRAM_DENOM_37)) {
            return new V0();
        }

        return new V1(format);
    }

    private static class V0 implements NakedSerializer<SnapshotCommandContent> {
        @Override
        public SnapshotCommandContent deserializeToEofImpl(StockpileDeserializer deserializer) {
            return new SnapshotCommandContent(new DeletedShardSet());
        }

        @Override
        public void serializeToEof(SnapshotCommandContent content, StockpileSerializer serializer) {
            // no op
        }
    }

    private static class V1 implements NakedSerializer<SnapshotCommandContent> {
        private final NakedSerializer<DeletedShardSet> deletedShardSerializer;

        public V1(StockpileFormat format) {
            this.deletedShardSerializer = DeletedShardSetSerializer.serializerForFormat(format);
        }

        @Override
        public void serializeToEof(SnapshotCommandContent content, StockpileSerializer serializer) {
            // (1) deletedShards
            deletedShardSerializer.serializeToEof(content.deletedShards(), serializer);
        }

        @Override
        public SnapshotCommandContent deserializeToEofImpl(StockpileDeserializer deserializer) {
            // (1) deleted shards
            var deletedShards = deletedShardSerializer.deserializeToEof(deserializer);
            return new SnapshotCommandContent(deletedShards);
        }
    }
}
