package ru.yandex.stockpile.server.data.dao;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;

import static ru.yandex.kikimr.client.kv.KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE;


/**
 * Sequentially writes (with inFlight=1) chunks of given file content.
 *
 * @author Sergey Polovko
 */
public final class LargeFileWriter {

    private final KikimrKvClientCounting kvClient;
    private final long tabletId;
    private final long gen;
    private final WriteClass writeClass;
    private final MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel;
    private final MsgbusKv.TKeyValueRequest.EPriority priority;

    public LargeFileWriter(
        KikimrKvClientCounting kvClient,
        long tabletId,
        long gen,
        WriteClass writeClass,
        MsgbusKv.TKeyValueRequest.EStorageChannel storageChannel,
        MsgbusKv.TKeyValueRequest.EPriority priority)
    {
        this.kvClient = kvClient;
        this.tabletId = tabletId;
        this.gen = gen;
        this.writeClass = writeClass;
        this.storageChannel = storageChannel;
        this.priority = priority;
    }

    public CompletableFuture<Void> write(String name, byte[] content) {
        int count = (content.length + DO_NOT_EXCEED_FILE_SIZE - 1) / DO_NOT_EXCEED_FILE_SIZE;

        List<String> names = IntStream.range(0, count)
            .mapToObj(i -> StockpileKvNames.TMP_PREFIX + "lg." + String.format("%03d.", i) + name)
            .collect(Collectors.toList());

        CompletableFuture<Void> writeFuture = new CompletableFuture<>();
        writeChunk(writeFuture, names, 0, content);

        return writeFuture.thenCompose(unit -> {
            return kvClient.concatAndDeleteOriginals(writeClass, tabletId, gen, names, name);
        });
    }

    private void writeChunk(CompletableFuture<Void> future, List<String> names, int index, byte[] content) {
        if (index >= names.size()) {
            future.complete(null);
            return;
        }

        try {
            String partName = names.get(index);
            int offset = index * DO_NOT_EXCEED_FILE_SIZE;
            int length = Math.min(DO_NOT_EXCEED_FILE_SIZE, content.length - offset);
            ByteString partBytes = ByteStringsStockpile.unsafeWrap(content, offset, length);

            kvClient.write(writeClass, tabletId, gen, partName, partBytes, storageChannel, priority)
                .whenComplete((response, throwable) -> {
                    if (throwable != null) {
                        future.completeExceptionally(throwable);
                    } else {
                        writeChunk(future, names, index + 1, content);
                    }
                });
        } catch (Throwable t) {
            future.completeExceptionally(t);
        }
    }
}
