package ru.yandex.stockpile.server.data.dao;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.concurrency.limits.actors.LimiterActorRunner;
import ru.yandex.concurrency.limits.actors.LimiterImpl;
import ru.yandex.concurrency.limits.actors.OperationProvider;
import ru.yandex.concurrency.limits.actors.OperationStatus;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.server.shard.ExceptionHandler;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ReadBatcher {
    @WillNotClose
    private final Executor executor;
    private final KikimrKvClientCounting kikimrKvClient;
    private final ReadBatcherMetrics metrics;
    private final MetricRegistry registry;
    final int inflightLimit;
    private final int inflightLimitPerTablet;

    private final ConcurrentLinkedDeque<Request> queue = new ConcurrentLinkedDeque<>();
    private final Long2ObjectOpenHashMap<TabletReader> readerPerTable = new Long2ObjectOpenHashMap<>();

    private final LimiterActorRunner actor;

    public ReadBatcher(
        @WillNotClose ExecutorService executor,
        KikimrKvClientCounting kikimrKvClient,
        int inFlightLimit,
        int inflightLimitPerTablet,
        MetricRegistry registry)
    {
        this.actor = LimiterActorRunner.newBuilder()
                .limiter(LimiterImpl.newBuilder()
                        .initLimit(1)
                        .minLimit(1)
                        .maxLimit(inFlightLimit)
                        .minRtt(TimeUnit.MICROSECONDS.toNanos(100))
                        .operation("batch_disk_read")
                        .registry(registry)
                        .build())
                .executor(executor)
                .operationProvider(new HostOperationProvider())
                .operation("batch_disk_read")
                .build();
        this.kikimrKvClient = kikimrKvClient;
        this.metrics = new ReadBatcherMetrics(registry);
        this.inflightLimit = inFlightLimit;
        this.inflightLimitPerTablet = inflightLimitPerTablet;
        this.executor = executor;
        this.registry = registry;
    }

    /**
     * @return empty arrays for missing files
     */
    public CompletableFuture<byte[][]> kvReadDataMulti(long tabletId, long gen, KvChunkAddress[] addresses) {
        return kvReadDataMulti(
            tabletId, gen, addresses,
            ThreadLocalTimeout.deadlineMillis());
    }

    public CompletableFuture<byte[]> kvReadData(long tabletId, long gen, KvChunkAddress address) {
        return kvReadDataMulti(tabletId, gen, new KvChunkAddress[] { address })
            .thenApply(r -> Arrays.stream(r).collect(CollectorsF.single()));
    }

    public CompletableFuture<byte[][]> kvReadDataMulti(long tabletId, long gen, KvChunkAddress[] addresses, long expiredAt) {
        var req = new Request(tabletId, gen, addresses, expiredAt, System.nanoTime(), new CompletableFuture<>());
        enqueue(req);
        return req.future;
    }

    private void enqueue(Request req) {
        metrics.inQueue.add(req.addresses.length);
        actor.addQueueSize(1);
        queue.addLast(req);
        actor.schedule();
    }

    private TabletReader tabletReader(long tabletId) {
        var result = readerPerTable.get(tabletId);
        if (result == null) {
            result = new TabletReader(tabletId, inflightLimitPerTablet, kikimrKvClient, executor, registry);
            readerPerTable.put(tabletId, result);
        }
        return result;
    }

    public CompletableFuture<Void> flushQueueForTablet(long tabletId) {
        var doneFuture = new CompletableFuture<Void>();
        // attach to oldest request in queue, because queue processed from tail
        for (Request req : queue) {
            if (req.tabletId == tabletId) {
                req.future.whenComplete((ignore, e) -> {
                    doneFuture.complete(null);
                });
                return doneFuture;
            }
        }

        return completedFuture(null);
    }

    private record Request(long tabletId, long gen, KvChunkAddress[] addresses, long expiredAt, long createdAtNanos, CompletableFuture<byte[][]> future) {
    }

    private class HostOperationProvider implements OperationProvider {
        @Override
        public boolean hasNext() {
            return !queue.isEmpty();
        }

        @Override
        public CompletableFuture<OperationStatus> next() {
            Request req;
            long expireThreshold = System.currentTimeMillis() + TimeUnit.MILLISECONDS.toMillis(500);
            long nowNanos = System.nanoTime();
            while ((req = queue.pollLast()) != null) {
                metrics.started.add(req.addresses.length);
                actor.addQueueTime(TimeUnit.NANOSECONDS.toMillis(nowNanos - req.createdAtNanos));
                actor.addQueueSize(-1);
                metrics.inQueue.add(-req.addresses.length);
                if (req.expiredAt <= expireThreshold) {
                    metrics.timeout.add(req.addresses.length);
                    actor.addStatus(OperationStatus.DROP);
                    req.future.completeExceptionally(new StockpileRuntimeException(EStockpileStatusCode.DEADLINE_EXCEEDED, "Deadline exceeded on read from tablet", false));
                    continue;
                }

                return startRead(req);
            }

            return completedFuture(OperationStatus.IGNORE);
        }

        private CompletableFuture<OperationStatus> startRead(Request req) {
            var startNanos = System.nanoTime();
            var reader = tabletReader(req.tabletId);
            var readFuture = CompletableFutures.safeCall(() -> {
                return reader.read(req.gen, req.addresses, req.expiredAt);
            });
            CompletableFutures.whenComplete(readFuture, req.future);
            return readFuture.handle((ignore, e) -> {
                long elapsedTimeMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                metrics.elapsedTimeMillis.record(elapsedTimeMs, req.addresses.length);
                if (ExceptionHandler.isGenerationChanged(e)) {
                    metrics.failed.add(req.addresses.length);
                    return OperationStatus.IGNORE;
                } else if (e != null) {
                    metrics.failed.add(req.addresses.length);
                    return OperationStatus.DROP;
                }

                metrics.completed.add(req.addresses.length);
                return OperationStatus.SUCCESS;
            });
        }
    }
}
