package ru.yandex.stockpile.server.data.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import ru.yandex.concurrency.limits.actors.LimiterActorRunner;
import ru.yandex.concurrency.limits.actors.LimiterImpl;
import ru.yandex.concurrency.limits.actors.OperationProvider;
import ru.yandex.concurrency.limits.actors.OperationStatus;
import ru.yandex.kikimr.client.kv.KvChunkAddress;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.server.shard.ExceptionHandler;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class TabletReader {
    // TODO: limited queue size (@gordiychuk)
    private final Queue<Request> queue = new ConcurrentLinkedQueue<>();

    private final long tabletId;
    private final KikimrKvClientCounting kvClient;
    private final LimiterActorRunner actor;

    public TabletReader(long tabletId, int maxInflight, KikimrKvClientCounting kvClient, Executor executor, MetricRegistry registry) {
        this.tabletId = tabletId;
        this.kvClient = kvClient;
        this.actor = LimiterActorRunner.newBuilder()
                .executor(executor)
                .limiter(LimiterImpl.newBuilder()
                        .initLimit(1)
                        .minLimit(1)
                        .maxLimit(maxInflight)
                        .operation("tablet_disk_read")
                        .minRtt(TimeUnit.MILLISECONDS.toNanos(100))
                        .registry(registry)
                        .build())
                .operationProvider(new TabletReadOperationProvider())
                .operation("tablet_disk_read")
                .build();
    }

    public CompletableFuture<byte[][]> read(long gen, KvChunkAddress[] addresses, long expiredAt) {
        List<Request> requests = new ArrayList<>(addresses.length);
        List<CompletableFuture<byte[]>> futures = new ArrayList<>(addresses.length);
        long createNanos = System.nanoTime();
        for (var address : addresses) {
            var req = new Request(gen, address, expiredAt, createNanos, new CompletableFuture<>());
            requests.add(req);
            futures.add(req.future);
        }
        enqueue(requests);
        return CompletableFutures.allOf(futures).thenApply(list -> list.toArray(new byte[0][]));
    }

    public CompletableFuture<byte[]> read(long gen, KvChunkAddress address, long expiredAt) {
        var req = new Request(gen, address, expiredAt, System.nanoTime(), new CompletableFuture<>());
        enqueue(List.of(req));
        return req.future;
    }

    private void enqueue(List<Request> requests) {
        actor.addQueueSize(requests.size());
        queue.addAll(requests);
        actor.schedule();
    }

    private static record Request(
            long gen,
            KvChunkAddress address,
            long expiredAt,
            long createdAtNanos,
            CompletableFuture<byte[]> future)
    {
    }

    private class TabletReadOperationProvider implements OperationProvider {
        @Override
        public boolean hasNext() {
            return !queue.isEmpty();
        }

        @Override
        public CompletableFuture<OperationStatus> next() {
            Request req;
            long expireThreshold = System.currentTimeMillis() + TimeUnit.MILLISECONDS.toMillis(500);
            long nowNanos = System.nanoTime();
            while ((req = queue.poll()) != null) {
                actor.addQueueTime(TimeUnit.NANOSECONDS.toMillis(nowNanos - req.createdAtNanos));
                actor.addQueueSize(-1);
                if (req.expiredAt <= expireThreshold) {
                    actor.addStatus(OperationStatus.DROP);
                    req.future.completeExceptionally(new StockpileRuntimeException(EStockpileStatusCode.DEADLINE_EXCEEDED, "Deadline exceeded on read from tablet", false));
                    continue;
                }

                return startRead(req);
            }

            return completedFuture(OperationStatus.IGNORE);
        }

        private CompletableFuture<OperationStatus> startRead(Request request) {
            var readFuture = CompletableFutures.safeCall(() -> kvClient.readDataLarge(ReadClass.SERVE_READ, tabletId, request.gen, request.address));
            CompletableFutures.whenComplete(readFuture, request.future);
            return readFuture.handle((ignore, e) -> {
                if (ExceptionHandler.isGenerationChanged(e)) {
                    return OperationStatus.IGNORE;
                } else if (e != null) {
                    return OperationStatus.DROP;
                }

                return OperationStatus.SUCCESS;
            });
        }
    }
}
