package ru.yandex.stockpile.server.data.index;

import java.util.Arrays;
import java.util.StringJoiner;
import java.util.stream.LongStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.util.collection.array.UnsignedLongArrays;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.server.data.chunk.ChunkIndex;
import ru.yandex.stockpile.server.data.chunk.ChunkIndexEntry;
import ru.yandex.stockpile.server.data.chunk.DataRangeInSnapshot;

/**
 * @author Vladimir Gordiychuk
 */
public class ChunkIndexArray implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ChunkIndexArray.class);

    @Nonnull
    private long[] localIdsSorted;
    @Nonnull
    private long[] lastTssMillis;
    @Nonnull
    private int[] nextOffsets; // element <i> stores offset for next item <i+1>
    private int[] nextChunkIndex;
    private int metricCount = 0;
    private int chunksCount = 0;

    public ChunkIndexArray() {
        this.localIdsSorted = Cf.LongArray.emptyArray();
        this.lastTssMillis = Cf.LongArray.emptyArray();
        this.nextOffsets = Cf.IntegerArray.emptyArray();
        this.nextChunkIndex = Cf.IntegerArray.emptyArray();
        this.chunksCount = 0;
    }

    public ChunkIndexArray(ChunkIndex[] chunks) {
        this.nextChunkIndex = new int[chunks.length];
        for (int index = 0; index < chunks.length; index++) {
            var chunk = chunks[index];
            metricCount += chunk.metricCount();
            nextChunkIndex[index] = metricCount;
        }
        this.chunksCount = chunks.length;
        this.localIdsSorted = new long[metricCount];
        this.lastTssMillis = new long[metricCount];
        this.nextOffsets = new int[metricCount];

        int pos = 0;
        for (var chunk : chunks) {
            int size = chunk.metricCount();
            System.arraycopy(chunk.getLocalIdsSortedArray(), 0, localIdsSorted, pos, size);
            System.arraycopy(chunk.getLastTssMillisArray(), 0, lastTssMillis, pos, size);
            System.arraycopy(chunk.getNextOffsetsArray(), 0, nextOffsets, pos, size);
            pos += size;
        }
    }

    public void addMetric(long localId, long lastTsMillis, int metricDataSize) {
        if (metricCount > 0) {
            long prevLocalId = localIdsSorted[metricCount - 1];
            if (StockpileLocalId.compare(localId, prevLocalId) <= 0) {
                throw new IllegalArgumentException("not sorted, " + StockpileLocalId.toString(prevLocalId) + " >= " + StockpileLocalId.toString(localId));
            }
        }

        if (metricCount == localIdsSorted.length) {
            int newCapacity = metricCount + (metricCount >>> 1) + 10;
            localIdsSorted = Arrays.copyOf(localIdsSorted, newCapacity);
            lastTssMillis = Arrays.copyOf(lastTssMillis, newCapacity);
            nextOffsets = Arrays.copyOf(nextOffsets, newCapacity);
        }

        localIdsSorted[metricCount] = localId;
        lastTssMillis[metricCount] = lastTsMillis;
        if (metricsAtLastChunk() == 0) {
            nextOffsets[metricCount] = metricDataSize;
        } else {
            nextOffsets[metricCount] = nextOffsets[metricCount - 1] + metricDataSize;
        }
        metricCount++;
    }

    public void finishChunk() {
        if (chunksCount == nextChunkIndex.length) {
            int newCapacity = chunksCount + (chunksCount >>> 1) + 10;
            nextChunkIndex = Arrays.copyOf(nextChunkIndex, newCapacity);
        }

        if (metricsAtLastChunk() > 0) {
            nextChunkIndex[chunksCount++] = metricCount;
        }
    }

    public void shrinkToFit() {
        if (metricCount != localIdsSorted.length) {
            localIdsSorted = Arrays.copyOf(localIdsSorted, metricCount);
            lastTssMillis = Arrays.copyOf(lastTssMillis, metricCount);
            nextOffsets = Arrays.copyOf(nextOffsets, metricCount);
        }

        if (chunksCount != nextChunkIndex.length) {
            nextChunkIndex = Arrays.copyOf(nextChunkIndex, chunksCount);
        }
    }

    @Nullable
    public DataRangeInSnapshot findMetricData(long localId) {
        var metric = findMetric(localId);
        if (metric == null) {
            return null;
        }

        return metric.getDataRange();
    }

    @Nullable
    public ChunkIndexEntry findMetric(long localId) {
        int metricIndex = findMetricIndex(localId);
        if (metricIndex < 0) {
            return null;
        }

        int chunkNo = findChunkByIndex(metricIndex);
        if (chunkNo < 0) {
            return null;
        }

        final int offset;
        final int length;
        if (metricIndex == chunkFromIndex(chunkNo)) {
            offset = 0;
            length = nextOffsets[metricIndex];
        } else {
            offset = nextOffsets[metricIndex - 1];
            length = nextOffsets[metricIndex] - nextOffsets[metricIndex - 1];
        }

        return new ChunkIndexEntry(chunkNo, localId, lastTssMillis[metricIndex], offset, length);
    }

    public ChunkIndex[] chunks() {
        ChunkIndex[] result = new ChunkIndex[chunksCount];
        for (int chunkNo = 0; chunkNo < chunksCount; chunkNo++) {
            result[chunkNo] = getChunk(chunkNo);
        }
        return result;
    }

    public ChunkIndex getChunk(int chunkNo) {
        int from = chunkFromIndex(chunkNo);
        int to = nextChunkIndex[chunkNo];
        long[] localIdsSorted = Arrays.copyOfRange(this.localIdsSorted, from, to);
        long[] lastTssMillis = Arrays.copyOfRange(this.lastTssMillis, from, to);
        int[] nextOffsets = Arrays.copyOfRange(this.nextOffsets, from, to);
        return new ChunkIndex(localIdsSorted, lastTssMillis, nextOffsets);
    }

    public int getChunksCount() {
        return chunksCount;
    }

    public int getMetricCount() {
        return metricCount;
    }

    public int getMetricCapacity() {
        return localIdsSorted.length;
    }

    public long countChunkDiskSize() {
        if (metricCount == 0) {
            return 0;
        }

        long size = 0;
        for (int index = 0; index < chunksCount; index++) {
            size += nextOffsets[nextChunkIndex[index] - 1];
        }
        return size;
    }

    public LongStream metricIdStream() {
        return Arrays.stream(localIdsSorted, 0, metricCount);
    }

    /**
     * @return last metric timestamp or {@code -1} if metric was not found.
     */
    public long findMetricLastTsMillis(long localId) {
        int index = findMetricIndex(localId);
        if (index >= 0) {
            return lastTssMillis[index];
        }
        return -1;
    }

    private int metricsAtLastChunk() {
        if (chunksCount == 0) {
            return metricCount;
        }

        int from = nextChunkIndex[chunksCount - 1];
        return metricCount - from;
    }

    private int chunkFromIndex(int chunkNo) {
        if (chunkNo == 0) {
            return 0;
        }

        return nextChunkIndex[chunkNo - 1];
    }

    private int findMetricIndex(long localId) {
        if (metricCount == 0) {
            return -1;
        }

        if (StockpileLocalId.compare(localId, localIdsSorted[0]) < 0
            || StockpileLocalId.compare(localId, localIdsSorted[metricCount - 1]) > 0)
        {
            return -1;
        }

        return UnsignedLongArrays.binarySearch(localIdsSorted, 0, metricCount, localId);
    }

    private int findChunkByIndex(int metricIndex) {
        int i = Arrays.binarySearch(nextChunkIndex, 0, chunksCount, metricIndex);
        return i < 0 ? -i - 1 : i + 1;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.arrayObjectSize(localIdsSorted);
        size += MemoryCounter.arrayObjectSize(lastTssMillis);
        size += MemoryCounter.arrayObjectSize(nextOffsets);
        size += MemoryCounter.arrayObjectSize(nextChunkIndex);
        return size;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", ChunkIndexArray.class.getSimpleName() + "[", "]")
            .add("metricCount=" + metricCount)
            .add("chunksCount=" + chunksCount)
            .add("chunksBytes=" + DataSize.shortString(countChunkDiskSize()))
            .toString();
    }
}
