package ru.yandex.stockpile.server.data.index;

import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.Validate;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkAddressGlobal;
import ru.yandex.stockpile.server.data.chunk.IndexEntry;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SnapshotIndex implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SnapshotIndex.class);

    @Nonnull
    private final SnapshotLevel level;
    private final long txn;
    @Nonnull
    private final SnapshotIndexContent content;

    public SnapshotIndex(SnapshotLevel level, long txn, SnapshotIndexContent content) {
        Validate.isTrue(txn >= 0);

        this.level = level;
        this.txn = txn;
        this.content = content;
    }

    public boolean isEmpty() {
        return content.isEmpty();
    }

    @Nonnull
    public SnapshotLevel getLevel() {
        return level;
    }

    public long getTxn() {
        return txn;
    }

    public boolean isReal() {
        return txn > 0;
    }

    @Nonnull
    public SnapshotAddress snapshotAddress() {
        return new SnapshotAddress(level, txn);
    }

    @Nonnull
    public ChunkAddressGlobal[] chunkAddresses() {
        return IntStream.range(0, content.getChunksCount())
            .mapToObj(i -> new ChunkAddressGlobal(level, txn, i))
            .toArray(ChunkAddressGlobal[]::new);
    }

    @Nonnull
    public SnapshotIndexContent getContent() {
        return content;
    }

    @Nullable
    public IndexEntry findMetric(long localId) {
        var chunk = content.findMetric(localId);
        if (chunk == null) {
            return null;
        }

        return new IndexEntry(content.getFormat(), level, txn, chunk);
    }

    @Override
    public String toString() {
        return "SnapshotIndex{" +
            "level=" + level +
            ", txn=" + txn +
            ", content=" + content +
            '}';
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + content.memorySizeIncludingSelf();
    }
}
