package ru.yandex.stockpile.server.data.index;

import java.time.Instant;
import java.util.stream.LongStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.server.data.chunk.ChunkIndex;
import ru.yandex.stockpile.server.data.chunk.ChunkIndexEntry;
import ru.yandex.stockpile.server.data.chunk.DataRangeInSnapshot;
import ru.yandex.stockpile.server.data.index.stats.IndexStatsProject;
import ru.yandex.stockpile.server.shard.SnapshotReason;
import ru.yandex.stockpile.server.shard.stat.SizeAndCount;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class SnapshotIndexContent implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(SnapshotIndexContent.class);

    @Nonnull
    final StockpileFormat format;
    @InstantMillis
    final long tsMillis;
    @InstantMillis
    final long decimatedAt;
    @InstantMillis
    final long createDurationMillis;
    final long recordCount;

    private final IndexStatsProject stats;

    final long metricCount;

    @Nonnull
    final SnapshotReason snapshotReason;
    final ChunkIndexArray chunks;

    public SnapshotIndexContent(StockpileFormat format, SnapshotIndexProperties properties, ChunkIndex[] chunks) {
        this(format, properties, new ChunkIndexArray(chunks));
    }

    public SnapshotIndexContent(StockpileFormat format, SnapshotIndexProperties properties, ChunkIndexArray chunks) {
        this.format = format;
        this.tsMillis = properties.getTsMillis();
        this.decimatedAt = properties.getDecimatedAt();
        this.createDurationMillis = properties.getCreateDurationMillis();
        this.recordCount = properties.getRecordCount();
        this.snapshotReason = properties.getSnapshotReason();
        this.chunks = chunks;

        if (tsMillis == 0) {
            if (chunks.getChunksCount() != 0) {
                throw new IllegalArgumentException();
            }
        } else {
            InstantUtils.sanityCheckMillis(tsMillis);
        }

        this.stats = properties.getStatsByProject();
        this.metricCount = properties.metricCount;
    }

    public SnapshotIndexContent() {
        this(StockpileFormat.CURRENT, SnapshotIndexProperties.empty, new ChunkIndex[0]);
    }

    @Nonnull
    public SnapshotIndexProperties properties() {
        return new SnapshotIndexProperties()
            .setCreatedAt(tsMillis)
            .setCreateDurationMillis(createDurationMillis)
            .setDecimatedAt(decimatedAt)
            .setRecordCount(recordCount)
            .setSnapshotReason(snapshotReason)
            .setStatsByProject(stats)
            .setMetricCount(metricCount);
    }

    @Nonnull
    public SnapshotReason getSnapshotReason() {
        return snapshotReason;
    }

    public long getRecordCount() {
        return recordCount;
    }

    public long getMetricCount() {
        return metricCount;
    }

    public boolean isEmpty() {
        return chunks.getChunksCount() == 0;
    }

    public long getTsMillis() {
        return tsMillis;
    }

    public long getDecimatedAt() {
        return decimatedAt;
    }

    public ChunkIndex[] getChunks() {
        return chunks.chunks();
    }

    public ChunkIndex getChunk(int chunkNo) {
        return chunks.getChunk(chunkNo);
    }

    public int getChunksCount() {
        return chunks.getChunksCount();
    }

    public SizeAndCount diskSize() {
        return new SizeAndCount(chunks.countChunkDiskSize(), chunks.getChunksCount());
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += stats.memorySizeIncludingSelf();
        size += chunks.memorySizeIncludingSelf();
        return size;
    }

    public LongStream metricIdStream() {
        return chunks.metricIdStream();
    }

    @Nullable
    public DataRangeInSnapshot findMetricData(long localId) {
        return chunks.findMetricData(localId);
    }

    @Nullable
    public ChunkIndexEntry findMetric(long localId) {
        return chunks.findMetric(localId);
    }

    /**
     * @return last metric timestamp or {@code -1} if metric was not found.
     */
    public long findMetricLastTsMillis(long localId) {
        return chunks.findMetricLastTsMillis(localId);
    }

    @Nonnull
    public StockpileFormat getFormat() {
        return format;
    }

    public IndexStatsProject getStats() {
        return stats;
    }

    @Override
    public String toString() {
        return "SnapshotIndexContent{" +
            "version=" + format +
            ", ts=" + Instant.ofEpochMilli(tsMillis) +
            ", chunks=" + chunks.getChunksCount() +
            '}';
    }
}
