package ru.yandex.stockpile.server.data.index.stats;

import java.util.HashMap;
import java.util.Map;

import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.server.SnapshotLevel;

/**
 * @author Vladimir Gordiychuk
 */
public class IndexStatsLevel implements MemMeasurable, MetricSupplier {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IndexStatsLevel.class);

    @ProtoField(n = 1)
    private Map<SnapshotLevel, IndexStatsProject> byLevel;
    @ProtoField(n = 2)
    private IndexStatsProject total;

    public IndexStatsLevel() {
        byLevel = new HashMap<>();
        total = new IndexStatsProject();
    }

    public void combine(IndexStatsLevel stats) {
        for (var entry : stats.byLevel.entrySet()) {
            getStatsByLevel(entry.getKey()).combine(entry.getValue());
        }
        this.total.combine(stats.total);
    }

    public void add(SnapshotLevel level, MetricArchiveGeneric archive) {
        getStatsByLevel(level).add(archive);
        total.add(archive);
    }

    public void add(SnapshotLevel level, IndexStatsProject stats) {
        getStatsByLevel(level).combine(stats);
        total.combine(stats);
    }

    public IndexStatsProject getStatsByLevel(SnapshotLevel level) {
        return byLevel.computeIfAbsent(level, ignore -> new IndexStatsProject());
    }

    public Map<SnapshotLevel, IndexStatsProject> getStats() {
        return byLevel;
    }

    public IndexStatsProject getTotalByLevel() {
        return total;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.hashMapSize(byLevel);
        for (var stats : byLevel.values()) {
            size += stats.memorySizeIncludingSelf();
        }
        size += total.memorySizeIncludingSelf();
        return size;
    }

    @Override
    public int estimateCount() {
        return total.estimateCount() * (byLevel.size() + 1);
    }

    @Override
    public void append(long tsMillis, Labels labels, MetricConsumer consumer) {
        total.append(tsMillis, labels.add("level", "total"), consumer);
        for (var entry : byLevel.entrySet()) {
            entry.getValue().appendAggregate(tsMillis, labels.add("level", entry.getKey().name()), consumer);
        }
    }
}
