package ru.yandex.stockpile.server.data.index.stats;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializerImpl;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
@NakedSerializerImpl(SerializerForIndexStatsOwner.class)
public class IndexStatsOwner implements MemMeasurable, MetricSupplier {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IndexStatsOwner.class);

    private Int2ObjectMap<IndexStatsType> byOwner;
    private IndexStatsType total;

    public IndexStatsOwner() {
        byOwner = new Int2ObjectOpenHashMap<>();
        total = new IndexStatsType();
    }

    public IndexStatsOwner(Int2ObjectMap<IndexStatsType> byOwner) {
        this.byOwner = byOwner;
        this.total = new IndexStatsType();
        for (var value : byOwner.values()) {
            total.combine(value);
        }
    }

    public void combine(IndexStatsOwner stats) {
        for (var entry : stats.byOwner.int2ObjectEntrySet()) {
            getStatsByOwnerId(entry.getIntKey()).combine(entry.getValue());
        }
        this.total.combine(stats.total);
    }

    public Int2ObjectMap<IndexStatsType> getByOwner() {
        return byOwner;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.hashMapSize(byOwner);
        size += total.memorySizeIncludingSelf() * (byOwner.size() + 1);
        return size;
    }

    @Override
    public int estimateCount() {
        return total.estimateCount() * (byOwner.size() + 1);
    }

    public void add(MetricArchiveGeneric archive) {
        getStatsByOwnerId(archive.getOwnerShardId()).add(archive);
        total.add(archive);
    }

    public IndexStatsType getStatsByOwnerId(int ownerId) {
        var stats = byOwner.get(ownerId);
        if (stats == null) {
            stats = new IndexStatsType();
            byOwner.put(ownerId, stats);
        }
        return stats;
    }

    public IndexStatsData getTotalByKinds() {
        return total.getTotalByKinds();
    }

    public void appendAggregate(long tsMillis, Labels labels, MetricConsumer consumer) {
        total.append(tsMillis, labels.add("ownerId", "total"), consumer);
    }

    @Override
    public void append(long tsMillis, Labels labels, MetricConsumer consumer) {
        appendAggregate(tsMillis, labels, consumer);
        for (var entry : byOwner.int2ObjectEntrySet()) {
            entry.getValue().append(tsMillis, labels.add("ownerId", Integer.toUnsignedString(entry.getIntKey())), consumer);
        }
    }
}
