package ru.yandex.stockpile.server.data.index.stats;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.stockpile.api.EProjectId;

/**
 * @author Vladimir Gordiychuk
 */
public class IndexStatsProject implements MemMeasurable, MetricSupplier {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IndexStatsProject.class);

    @ProtoField(n = 2)
    private IndexStatsType total;
    @ProtoField(n = 3)
    @Nullable
    private Map<EProjectId, IndexStatsOwner> byProjectOwner;

    public IndexStatsProject() {
        byProjectOwner = new HashMap<>();
        total = new IndexStatsType();
    }

    public void combine(IndexStatsProject stats) {
        if (stats.byProjectOwner != null) {
            for (var entry : stats.byProjectOwner.entrySet()) {
                getStatsByProject(entry.getKey()).combine(entry.getValue());
            }
        }
        this.total.combine(stats.total);
    }

    public void add(MetricArchiveGeneric archive) {
        getStatsByProject(archive.getOwnerProjectIdOrUnknown()).add(archive);
        total.add(archive);
    }

    public IndexStatsOwner getStatsByProject(EProjectId projectId) {
        if (byProjectOwner == null) {
            byProjectOwner = new HashMap<>();
        }
        return byProjectOwner.computeIfAbsent(projectId, ignore -> new IndexStatsOwner());
    }

    public IndexStatsType getTotalByProjects() {
        return total;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        long totalSize = total.memorySizeIncludingSelf();
        if (byProjectOwner != null) {
            size += MemoryCounter.hashMapSize(byProjectOwner) + byProjectOwner.size() * totalSize;
        }
        size += totalSize;
        return size;
    }

    @Override
    public int estimateCount() {
        int size = byProjectOwner != null ? byProjectOwner.size() : 0;
        return total.estimateCount() * (size + 1);
    }

    public void appendAggregate(long tsMillis, Labels labels, MetricConsumer consumer) {
        total.append(tsMillis, labels.addAll(Labels.of("projectId", "total", "ownerId", "total")), consumer);
        if (byProjectOwner != null && !byProjectOwner.isEmpty()) {
            for (var entry : byProjectOwner.entrySet()) {
                entry.getValue().appendAggregate(tsMillis, labels.add("projectId", entry.getKey().name()), consumer);
            }
        }
    }

    @Override
    public void append(long tsMillis, Labels labels, MetricConsumer consumer) {
        total.append(tsMillis, labels.addAll(Labels.of("projectId", "total", "ownerId", "total")), consumer);
        if (byProjectOwner != null && !byProjectOwner.isEmpty()) {
            for (var entry : byProjectOwner.entrySet()) {
                entry.getValue().append(tsMillis, labels.add("projectId", entry.getKey().name()), consumer);
            }
        }
    }
}
