package ru.yandex.stockpile.server.data.index.stats;

import java.util.HashMap;
import java.util.Map;
import java.util.StringJoiner;

import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.protobuf.MetricType;

/**
 * @author Vladimir Gordiychuk
 */
public class IndexStatsType implements MemMeasurable, MetricSupplier {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(IndexStatsType.class);

    @ProtoField(n = 1)
    private Map<MetricType, IndexStatsData> byType;
    @ProtoField(n = 2)
    private IndexStatsData total;

    public IndexStatsType() {
        this.byType = new HashMap<>();
        this.total = new IndexStatsData();
    }

    public void combine(IndexStatsType stats) {
        for (var entry : stats.byType.entrySet()) {
            getStatsByKind(entry.getKey()).combine(entry.getValue());
        }
        this.total.combine(stats.total);
    }

    public void add(MetricArchiveGeneric archive) {
        getStatsByKind(archive.getType()).add(archive);
        total.add(archive);
    }

    public IndexStatsData getStatsByKind(MetricType type) {
        return byType.computeIfAbsent(type, ignore -> new IndexStatsData());
    }

    public IndexStatsData getTotalByKinds() {
        return total;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += MemoryCounter.hashMapSize(byType);
        for (var stats : byType.values()) {
            size += stats.memorySizeIncludingSelf();
        }
        size += total.memorySizeIncludingSelf();
        return size;
    }

    @Override
    public int estimateCount() {
        return total.estimateCount() * (byType.size() + 1);
    }

    @Override
    public void append(long tsMillis, Labels labels, MetricConsumer consumer) {
        total.append(tsMillis, labels.add("kind", "total"), consumer);
        for (var entry : byType.entrySet()) {
            entry.getValue().append(tsMillis, labels.add("kind", entry.getKey().name()), consumer);
        }
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", IndexStatsType.class.getSimpleName() + "[", "]")
                .add("total=" + total)
                .toString();
    }
}
