package ru.yandex.stockpile.server.data.log;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.memState.EstimateSerializedSize;
import ru.yandex.stockpile.memState.MetricToArchiveMap;
import ru.yandex.stockpile.server.data.DeletedShardSet;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileLogEntryContent implements MemMeasurable, EstimateSerializedSize {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(StockpileLogEntryContent.class);

    /**
     * TODO: use {@link MetricToArchiveMap}
     */
    final Long2ObjectOpenHashMap<MetricArchiveMutable> dataByMetricId;
    final Int2LongMap producerSeqNoById;
    final DeletedShardSet deletedShards;

    public StockpileLogEntryContent() {
        this(new Long2ObjectOpenHashMap<>(), new DeletedShardSet(), new Int2LongOpenHashMap());
    }

    public StockpileLogEntryContent(Long2ObjectOpenHashMap<MetricArchiveMutable> dataByMetricId, DeletedShardSet deletedShards, Int2LongMap producerSeqNoById) {
        this.dataByMetricId = dataByMetricId;
        this.producerSeqNoById = producerSeqNoById;
        this.deletedShards = deletedShards;
    }

    @Nonnull
    public Long2ObjectOpenHashMap<MetricArchiveMutable> getDataByMetricId() {
        return dataByMetricId;
    }

    public Int2LongMap getProducerSeqNoById() {
        return producerSeqNoById;
    }

    public DeletedShardSet getDeletedShards() {
        return deletedShards;
    }

    public void overrideWith(StockpileLogEntryContent entry) {
        for (Long2ObjectMap.Entry<MetricArchiveMutable> e : entry.dataByMetricId.long2ObjectEntrySet()) {
            this.addMetricMultiArchive(e.getLongKey(), e.getValue());
        }
    }

    @Nonnull
    public MetricArchiveMutable archiveRefByLocalId(long localId) {
        MetricArchiveMutable archive = dataByMetricId.get(localId);
        if (archive == null) {
            archive = new MetricArchiveMutable();
            dataByMetricId.put(localId, archive);
        }
        return archive;
    }

    public void addMetric(long localId, AggrGraphDataIterable value) {
        MetricArchiveMutable archive = archiveRefByLocalId(localId);
        archive.addAll(value);
    }

    public void addMetricMultiArchive(long localId, MetricArchiveMutable archive) {
        MetricArchiveMutable prev = dataByMetricId.get(localId);
        if (prev == null) {
            dataByMetricId.put(localId, new MetricArchiveMutable(archive));
        } else {
            prev.updateWith(archive);
        }
    }

    public void addArchive(long localId, MetricArchiveMutable archive) {
        MetricArchiveMutable prev = dataByMetricId.get(localId);
        if (prev == null) {
            dataByMetricId.put(localId, archive);
        } else {
            prev.updateWith(archive);
        }
    }

    public void addPointForTest(long localId, long tsMillis, double value) {
        archiveRefByLocalId(localId).addRecord(AggrPoint.shortPoint(tsMillis, value));
    }

    public void addAggrPoint(long localId, AggrPoint aggrPoint) {
        MetricArchiveMutable archive = archiveRefByLocalId(localId);
        archive.addRecord(aggrPoint);
    }

    @Override
    public String toString() {
        return dataByMetricId.size() + " records, ...";
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE
                + MemoryCounter.long2ObjectOpenHashMapSizeWithContent(dataByMetricId)
                + MemoryCounter.int2LongMapSize(producerSeqNoById)
                + deletedShards.memorySizeIncludingSelf();
    }

    @Override
    public long estimateSerializedSize() {
        return memorySizeIncludingSelf() + dataByMetricId.size() * 5L;
    }

    public StockpileLogEntryProperties properties() {
        return new StockpileLogEntryProperties(dataByMetricId.size());
    }

    public int getMetricsCount() {
        return dataByMetricId.size();
    }

    public void release() {
        for (var archive : dataByMetricId.values()) {
            archive.close();
        }
        dataByMetricId.clear();
    }
}
