package ru.yandex.stockpile.server.data.log;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2LongMaps;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMaps;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveImmutableSelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.SelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.WithVersionHeaderSerializer;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.DeletedShardSetSerializer;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StockpileLogEntryContentImmutableSerializer extends WithVersionHeaderSerializer<StockpileLogEntryContentImmutable> {
    public static final NakedSerializer<StockpileLogEntryContentImmutable> S =
            new StockpileLogEntryContentImmutableSerializer(StockpileFormat.CURRENT);

    StockpileLogEntryContentImmutableSerializer(StockpileFormat format) {
        super(StockpileLogEntryContentImmutableSerializer::serializerForFormat, e -> format);
    }

    @Nonnull
    public static NakedSerializer<StockpileLogEntryContentImmutable> serializerForFormat(StockpileFormat format) {
        if (format.le(StockpileFormat.HISTOGRAM_DENOM_37)) {
            return new V1(format);
        } else if (format.le(StockpileFormat.IDEMPOTENT_WRITE_38)) {
            return new V2(format);
        }

        return new V3(format);
    }

    private static class V1 implements NakedSerializer<StockpileLogEntryContentImmutable> {

        private final SelfContainedSerializer<MetricArchiveImmutable> entrySerializer;
        private final StockpileLogEntryPropertiesSerializer logEntryPropertiesSer;

        public V1(StockpileFormat format) {
            entrySerializer = MetricArchiveImmutableSelfContainedSerializer.makeSerializerForVersion(format);
            logEntryPropertiesSer = StockpileLogEntryPropertiesSerializer.makeSerializerFor(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContentImmutable logEntryContent, StockpileSerializer serializer) {
            logEntryPropertiesSer.serializeWithLength(logEntryContent.properties(), serializer);
            for (Long2ObjectMap.Entry<MetricArchiveImmutable> e : logEntryContent.getDataByMetricId().long2ObjectEntrySet()) {
                serializer.writeFixed64(e.getLongKey());
                entrySerializer.serializeWithLength(e.getValue(), serializer);
            }
        }

        @Override
        public StockpileLogEntryContentImmutable deserializeToEofImpl(StockpileDeserializer deserializer) {
            StockpileLogEntryProperties properties = logEntryPropertiesSer.deserializeWithLength(deserializer);

            // consider parallel arrays or maybe even Long2ObjectArrayMap to free 25% additional space
            var archives = new Long2ObjectOpenHashMap<MetricArchiveImmutable>(properties.getArchivesCount());
            while (!deserializer.atEof()) {
                long metricId = deserializer.readFixed64();
                MetricArchiveImmutable archive = entrySerializer.deserializeWithLength(deserializer);
                archives.put(metricId, archive);
            }

            return new StockpileLogEntryContentImmutable(Long2ObjectMaps.unmodifiable(archives), new DeletedShardSet(), Int2LongMaps.EMPTY_MAP);
        }
    }

    private static class V2 implements NakedSerializer<StockpileLogEntryContentImmutable> {
        private final SelfContainedSerializer<MetricArchiveImmutable> archiveSerializer;

        public V2(StockpileFormat format) {
            archiveSerializer = MetricArchiveImmutableSelfContainedSerializer.makeSerializerForVersion(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContentImmutable log, StockpileSerializer serializer) {
            // (1) metrics
            var metrics = log.getDataByMetricId();
            serializer.writeVarint32(metrics.size());
            for (Long2ObjectMap.Entry<MetricArchiveImmutable> e : metrics.long2ObjectEntrySet()) {
                serializer.writeFixed64(e.getLongKey());
                archiveSerializer.serializeWithLength(e.getValue(), serializer);
            }

            // (2) producerSeqNo
            var producerSeqNoById = log.getProducerSeqNoById();
            serializer.writeVarint32(producerSeqNoById.size());
            for (var entry : producerSeqNoById.int2LongEntrySet()) {
                serializer.writeVarint32(entry.getIntKey());
                serializer.writeVarint64(entry.getLongValue());
            }
        }

        @Override
        public StockpileLogEntryContentImmutable deserializeToEofImpl(StockpileDeserializer deserializer) {
            // (1) metrics
            int metricsSize = deserializer.readVarint32();
            var metrics = new Long2ObjectOpenHashMap<MetricArchiveImmutable>(metricsSize);
            for (int index = 0; index < metricsSize; index++) {
                long metricId = deserializer.readFixed64();
                var archive = archiveSerializer.deserializeWithLength(deserializer);
                metrics.put(metricId, archive);
            }

            // (2) producerSeqNo
            int producersSize = deserializer.readVarint32();
            var producerSeqNoById = new Int2LongOpenHashMap(producersSize);
            for (int index = 0; index < producersSize; index++) {
                int producerId = deserializer.readVarint32();
                long producerSeqNo = deserializer.readVarint64();
                producerSeqNoById.put(producerId, producerSeqNo);
            }

            return new StockpileLogEntryContentImmutable(metrics, new DeletedShardSet(), producerSeqNoById);
        }
    }

    private static class V3 implements NakedSerializer<StockpileLogEntryContentImmutable> {
        private final SelfContainedSerializer<MetricArchiveImmutable> archiveSerializer;
        private final NakedSerializer<DeletedShardSet> deletedShardSerializer;

        public V3(StockpileFormat format) {
            archiveSerializer = MetricArchiveImmutableSelfContainedSerializer.makeSerializerForVersion(format);
            deletedShardSerializer = DeletedShardSetSerializer.serializerForFormat(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContentImmutable log, StockpileSerializer serializer) {
            // (1) metrics
            var metrics = log.getDataByMetricId();
            serializer.writeVarint32(metrics.size());
            for (Long2ObjectMap.Entry<MetricArchiveImmutable> e : metrics.long2ObjectEntrySet()) {
                serializer.writeFixed64(e.getLongKey());
                archiveSerializer.serializeWithLength(e.getValue(), serializer);
            }

            // (2) producerSeqNo
            var producerSeqNoById = log.getProducerSeqNoById();
            serializer.writeVarint32(producerSeqNoById.size());
            for (var entry : producerSeqNoById.int2LongEntrySet()) {
                serializer.writeVarint32(entry.getIntKey());
                serializer.writeVarint64(entry.getLongValue());
            }

            // (3) deletedShards
            deletedShardSerializer.serializeToEof(log.getDeletedShards(), serializer);
        }

        @Override
        public StockpileLogEntryContentImmutable deserializeToEofImpl(StockpileDeserializer deserializer) {
            // (1) metrics
            int metricsSize = deserializer.readVarint32();
            var metrics = new Long2ObjectOpenHashMap<MetricArchiveImmutable>(metricsSize);
            for (int index = 0; index < metricsSize; index++) {
                long metricId = deserializer.readFixed64();
                var archive = archiveSerializer.deserializeWithLength(deserializer);
                metrics.put(metricId, archive);
            }

            // (2) producerSeqNo
            int producersSize = deserializer.readVarint32();
            var producerSeqNoById = new Int2LongOpenHashMap(producersSize);
            for (int index = 0; index < producersSize; index++) {
                int producerId = deserializer.readVarint32();
                long producerSeqNo = deserializer.readVarint64();
                producerSeqNoById.put(producerId, producerSeqNo);
            }

            // (3) deleted shards
            var deletedShards = deletedShardSerializer.deserializeToEof(deserializer);
            return new StockpileLogEntryContentImmutable(metrics, deletedShards, producerSeqNoById);
        }
    }
}
