package ru.yandex.stockpile.server.data.log;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveMutableSelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.SelfContainedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.codec.serializer.StockpileSerializer;
import ru.yandex.solomon.codec.serializer.WithVersionHeaderSerializer;
import ru.yandex.solomon.codec.serializer.naked.NakedSerializer;
import ru.yandex.stockpile.server.data.DeletedShardSet;
import ru.yandex.stockpile.server.data.DeletedShardSetSerializer;

/**
 * @author Stepan Koltsov
 *
 * @see ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer
 */
@ParametersAreNonnullByDefault
public class StockpileLogEntryContentSerializer extends WithVersionHeaderSerializer<StockpileLogEntryContent> {
    public static final NakedSerializer<StockpileLogEntryContent> S =
        new StockpileLogEntryContentSerializer(StockpileFormat.CURRENT);

    StockpileLogEntryContentSerializer(StockpileFormat format) {
        super(StockpileLogEntryContentSerializer::serializerForFormat, e -> format);
    }

    @Nonnull
    public static NakedSerializer<StockpileLogEntryContent> serializerForFormat(StockpileFormat format) {
        if (format.le(StockpileFormat.HISTOGRAM_DENOM_37)) {
            return new V1(format);
        } else if (format.le(StockpileFormat.IDEMPOTENT_WRITE_38)) {
            return new V2(format);
        }

        return new V3(format);
    }

    private static class V1 implements NakedSerializer<StockpileLogEntryContent> {
        private final SelfContainedSerializer<MetricArchiveMutable> entrySerializer;
        private final StockpileLogEntryPropertiesSerializer logEntryPropertiesSer;

        public V1(StockpileFormat format) {
            entrySerializer = MetricArchiveMutableSelfContainedSerializer.makeSerializerForVersion(format);
            logEntryPropertiesSer = StockpileLogEntryPropertiesSerializer.makeSerializerFor(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContent logEntryContent, StockpileSerializer serializer) {
            logEntryPropertiesSer.serializeWithLength(logEntryContent.properties(), serializer);
            var it = logEntryContent.getDataByMetricId().long2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var e = it.next();
                serializer.writeFixed64(e.getLongKey());
                entrySerializer.serializeWithLength(e.getValue(), serializer);
            }
        }

        @Override
        public StockpileLogEntryContent deserializeToEofImpl(StockpileDeserializer deserializer) {
            StockpileLogEntryProperties properties = logEntryPropertiesSer.deserializeWithLength(deserializer);

            // consider parallel arrays or maybe even Long2ObjectArrayMap to free 25% additional space
            var archives = new Long2ObjectOpenHashMap<MetricArchiveMutable>(properties.getArchivesCount());
            while (!deserializer.atEof()) {
                long metricId = deserializer.readFixed64();
                MetricArchiveMutable archive = entrySerializer.deserializeWithLength(deserializer);
                archives.put(metricId, archive);
            }

            // TODO: use Long2ObjectMap interface and wrap archives with Long2ObjectMaps.unmodifiable()
            return new StockpileLogEntryContent(archives, new DeletedShardSet(), new Int2LongOpenHashMap());
        }
    }

    private static class V2 implements NakedSerializer<StockpileLogEntryContent> {
        private final SelfContainedSerializer<MetricArchiveMutable> archiveSerializer;

        public V2(StockpileFormat format) {
            archiveSerializer = MetricArchiveMutableSelfContainedSerializer.makeSerializerForVersion(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContent log, StockpileSerializer serializer) {
            // (1) metrics
            var metrics = log.getDataByMetricId();
            serializer.writeVarint32(metrics.size());
            var it = metrics.long2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var e = it.next();
                serializer.writeFixed64(e.getLongKey());
                archiveSerializer.serializeWithLength(e.getValue(), serializer);
            }

            // (2) producerSeqNo
            var producerSeqNoById = log.getProducerSeqNoById();
            serializer.writeVarint32(producerSeqNoById.size());
            for (var entry : producerSeqNoById.int2LongEntrySet()) {
                serializer.writeVarint32(entry.getIntKey());
                serializer.writeVarint64(entry.getLongValue());
            }
        }

        @Override
        public StockpileLogEntryContent deserializeToEofImpl(StockpileDeserializer deserializer) {
            // (1) metrics
            int metricsSize = deserializer.readVarint32();
            var metrics = new Long2ObjectOpenHashMap<MetricArchiveMutable>(metricsSize);
            for (int index = 0; index < metricsSize; index++) {
                long metricId = deserializer.readFixed64();
                var archive = archiveSerializer.deserializeWithLength(deserializer);
                metrics.put(metricId, archive);
            }

            // (2) producerSeqNo
            int producersSize = deserializer.readVarint32();
            var producerSeqNoById = new Int2LongOpenHashMap(producersSize);
            for (int index = 0; index < producersSize; index++) {
                int producerId = deserializer.readVarint32();
                long producerSeqNo = deserializer.readVarint64();
                producerSeqNoById.put(producerId, producerSeqNo);
            }

            return new StockpileLogEntryContent(metrics, new DeletedShardSet(), producerSeqNoById);
        }
    }

    private static class V3 implements NakedSerializer<StockpileLogEntryContent> {
        private final SelfContainedSerializer<MetricArchiveMutable> archiveSerializer;
        private final NakedSerializer<DeletedShardSet> deletedShardSerializer;

        public V3(StockpileFormat format) {
            archiveSerializer = MetricArchiveMutableSelfContainedSerializer.makeSerializerForVersion(format);
            deletedShardSerializer = DeletedShardSetSerializer.serializerForFormat(format);
        }

        @Override
        public void serializeToEof(StockpileLogEntryContent log, StockpileSerializer serializer) {
            // (1) metrics
            var metrics = log.getDataByMetricId();
            serializer.writeVarint32(metrics.size());
            var it = metrics.long2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var e = it.next();
                serializer.writeFixed64(e.getLongKey());
                archiveSerializer.serializeWithLength(e.getValue(), serializer);
            }

            // (2) producerSeqNo
            var producerSeqNoById = log.getProducerSeqNoById();
            serializer.writeVarint32(producerSeqNoById.size());
            for (var entry : producerSeqNoById.int2LongEntrySet()) {
                serializer.writeVarint32(entry.getIntKey());
                serializer.writeVarint64(entry.getLongValue());
            }

            // (3) deletedShards
            deletedShardSerializer.serializeToEof(log.deletedShards, serializer);
        }

        @Override
        public StockpileLogEntryContent deserializeToEofImpl(StockpileDeserializer deserializer) {
            // (1) metrics
            int metricsSize = deserializer.readVarint32();
            var metrics = new Long2ObjectOpenHashMap<MetricArchiveMutable>(metricsSize);
            for (int index = 0; index < metricsSize; index++) {
                long metricId = deserializer.readFixed64();
                var archive = archiveSerializer.deserializeWithLength(deserializer);
                metrics.put(metricId, archive);
            }

            // (2) producerSeqNo
            int producersSize = deserializer.readVarint32();
            var producerSeqNoById = new Int2LongOpenHashMap(producersSize);
            for (int index = 0; index < producersSize; index++) {
                int producerId = deserializer.readVarint32();
                long producerSeqNo = deserializer.readVarint64();
                producerSeqNoById.put(producerId, producerSeqNo);
            }

            // (3) deleted shards
            var deletedShards = deletedShardSerializer.deserializeToEof(deserializer);
            return new StockpileLogEntryContent(metrics, deletedShards, producerSeqNoById);
        }
    }
}
