package ru.yandex.stockpile.server.data.names;

import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.stockpile.server.data.names.file.ChunkFile;
import ru.yandex.stockpile.server.data.names.file.CommandFile;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;
import ru.yandex.stockpile.server.data.names.file.ProducerSeqNoFile;
import ru.yandex.stockpile.server.data.names.parserFormatter.Parser;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class FileNameParsed {

    @Nonnull
    public abstract FileKind fileKind();

    @Nonnull
    public abstract String reconstructWithinBackup();

    @Nonnull
    public String reconstruct(FileNamePrefix prefix) {
        return prefix.format() + reconstructWithinBackup();
    }

    @Nonnull
    public String reconstructCurrent() {
        return reconstruct(FileNamePrefix.Current.instance);
    }

    @Override
    public final String toString() {
        return "<b>." + reconstructWithinBackup();
    }

    // separate class to override initialization order
    private static class ParserHolder {
        private static final Parser<FileNameParsed> parser = Parser.or(
            LogFile.pf,
            IndexFile.pf,
            ChunkFile.pf,
            CommandFile.pf,
            ProducerSeqNoFile.pf
        );
    }

    private static Parser<FileNameParsed> parserInBackup(FileNamePrefix backupPrefix) {
        return ParserHolder.parser.addPrefix(backupPrefix.format());
    }

    public static FileNameParsed parseInBackup(FileNamePrefix backupPrefix, String fileName) {
        return parserInBackup(backupPrefix).parseOrThrow(fileName);
    }

    public static FileNameParsed parseCurrent(String fileName) {
        return parseInBackup(FileNamePrefix.Current.instance, fileName);
    }

    public static Optional<FileNameParsed> parseCurrentOptional(String fileName) {
        return parserInBackup(FileNamePrefix.Current.instance).parse(fileName);
    }
}
