package ru.yandex.stockpile.server.data.names;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.server.data.names.parserFormatter.Parser;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;
import ru.yandex.stockpile.server.data.names.parserFormatter.TwoWayFunction;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class FileNamePrefix {
    private FileNamePrefix() {
    }

    public abstract String format();

    @Override
    public int hashCode() {
        return format().hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }

        return format().equals(((FileNamePrefix) obj).format());
    }

    @Nonnull
    public NameRange toNameRange() {
        return StringMicroUtils.asciiPrefixToRange(format());
    }

    @Override
    public final String toString() {
        return format();
    }

    public boolean isCurrent() {
        return equals(Current.instance);
    }

    public static final class Current extends FileNamePrefix {
        private Current() {
        }

        public static final Current instance = new Current();

        public static final ParserFormatter<Void> pfUnit = ParserFormatters.str(StockpileKvNames.CURRENT_PREFIX);

        public static final ParserFormatter<Current> pf = pfUnit
            .map(new TwoWayFunction<Void, Current>() {
                @Override
                public Current thatWay(Void aVoid) {
                    return instance;
                }

                @Override
                public Void backwards(Current current) {
                    return null;
                }
            });

        @Override
        public String format() {
            return pf.format(this);
        }

        @Override
        public boolean equals(Object obj) {
            return obj instanceof Current;
        }

        @Override
        public int hashCode() {
            return Current.class.hashCode();
        }
    }

    public abstract static class ExceptCurrent extends FileNamePrefix {

    }

    public static final class Backup extends ExceptCurrent {
        private final long instantSeconds;

        public Backup(long instantSeconds) {
            InstantUtils.sanityCheckSeconds(instantSeconds);
            this.instantSeconds = instantSeconds;
        }

        private Instant instant() {
            return Instant.ofEpochSecond(instantSeconds);
        }

        public long getInstantSeconds() {
            return instantSeconds;
        }

        private static final DateTimeFormatter dateTimeFormatter =
            DateTimeFormatter.ofPattern("yyyyMMdd'T'HHmmss");

        public static final ParserFormatter<Backup> pf =
            ParserFormatters.fixedLengthString("20141122T121314".length())
                .addPrefix(StockpileKvNames.BACKUP_FIRST_LETTER)
                .addSuffix(".")
                .map(new TwoWayFunction<String, Backup>() {
                    @Override
                    public Backup thatWay(String s) {
                        Instant instant = LocalDateTime.parse(s, dateTimeFormatter).toInstant(ZoneOffset.UTC);
                        return new Backup(instant.getEpochSecond());
                    }

                    @Override
                    public String backwards(Backup backup) {
                        return backup.instant().atOffset(ZoneOffset.UTC).toLocalDateTime()
                            .format(dateTimeFormatter);
                    }
                });

        @Override
        public String format() {
            return pf.format(this);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Backup backup = (Backup) o;

            return instantSeconds == backup.instantSeconds;

        }

        @Override
        public int hashCode() {
            return Long.hashCode(instantSeconds);
        }
    }

    public static class TempBackup extends ExceptCurrent {
        private final Backup backup;

        public TempBackup(Backup backup) {
            this.backup = backup;
        }

        public static final ParserFormatter<TempBackup> pf = Backup.pf.addPrefix(StockpileKvNames.tmpPrefixPf)
            .map(new TwoWayFunction<Backup, TempBackup>() {
                @Override
                public TempBackup thatWay(Backup backup) {
                    return new TempBackup(backup);
                }

                @Override
                public Backup backwards(TempBackup tempBackup) {
                    return tempBackup.backup;
                }
            });

        @Override
        public String format() {
            return pf.format(this);
        }
    }

    private static class ParserHolder {
        private static final Parser<FileNamePrefix> parser = Parser.or(
            Current.pf,
            Backup.pf,
            TempBackup.pf
        );
    }

    @Nonnull
    public static FileNamePrefix parse(String p) {
        return ParserHolder.parser.parseOrThrow(p);
    }
}
