package ru.yandex.stockpile.server.data.names;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.collection.GroupByFlat;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.Txn;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class IndexAddressWithFileCount {
    private final SnapshotLevel level;
    private final long txn;
    private final int count;

    public IndexAddressWithFileCount(SnapshotLevel level, long txn, int count) {
        this.level = level;
        this.txn = txn;
        this.count = count;
    }

    @Nonnull
    public SnapshotLevel getLevel() {
        return level;
    }

    public long getTxn() {
        return txn;
    }

    public Txn getTxnObject() {
        return new Txn(txn);
    }

    public SnapshotAddress snapshotAddress() {
        return new SnapshotAddress(level, txn);
    }

    public int getCount() {
        return count;
    }

    public IndexFile[] indexFileNames() {
        return IntStream.range(0, count)
            .mapToObj(i -> new IndexFile(level, txn, i, i == count - 1))
            .toArray(IndexFile[]::new);
    }

    public static IndexAddressWithFileCount fold1(List<IndexFile> fileNamesSorted) {
        return NumberNamePart.fold1(
            fileNamesSorted,
            IndexFile::snapshotAddress,
            (a, i) -> new IndexAddressWithFileCount(a.level(), a.txn(), i));

    }

    public static List<IndexAddressWithFileCount> fold(List<IndexFile> indexFileNamesSorted) {
        return GroupByFlat.groupBy(indexFileNamesSorted, (a, b) -> a.snapshotAddress().equals(b.snapshotAddress()))
            .stream()
            .map(IndexAddressWithFileCount::fold1)
            .collect(Collectors.toList());
    }

    public static List<IndexAddressWithFileCount> filter(List<FileNameParsed> files) {
        List<IndexFile> indexFiles = files.stream()
            .filter(f -> f instanceof IndexFile)
            .map(f -> (IndexFile) f)
            .collect(Collectors.toList());
        return fold(indexFiles);
    }

    @Override
    public String toString() {
        return level.index.currentPrefix() + StockpileKvNames.txnPart.format(txn) + BigFilePart.countSuffixToString(count);
    }
}
