package ru.yandex.stockpile.server.data.names;

import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatter;
import ru.yandex.stockpile.server.data.names.parserFormatter.ParserFormatters;
import ru.yandex.stockpile.server.data.names.parserFormatter.TwoWayFunction;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class NumberNamePart {

    private final int digitCount;
    public final long max;

    public NumberNamePart(int digitCount) {
        if (digitCount < 0) {
            throw new IllegalArgumentException();
        }
        this.digitCount = digitCount;
        long max = 1;
        for (int i = 0; i < digitCount; ++i) {
            max *= 10;
        }
        this.max = max - 1;
    }

    public int getDigitCount() {
        return digitCount;
    }

    public String format(long n) {
        if (n < 0 || n > max) {
            throw new IllegalArgumentException();
        }

        String s = Long.toString(n);
        StringBuilder sb = new StringBuilder();
        while (sb.length() < digitCount - s.length()) {
            sb.append("0");
        }
        sb.append(s);
        return sb.toString();
    }

    public ParserFormatter<Long> parserFormatter() {
        return ParserFormatters.numberPart(this);
    }

    public ParserFormatter<Integer> integerParserFormatter() {
        return parserFormatter().map(new TwoWayFunction<Long, Integer>() {
            @Override
            public Integer thatWay(Long aLong) {
                return Math.toIntExact(aLong);
            }

            @Override
            public Long backwards(Integer integer) {
                return integer.longValue();
            }
        });
    }


    public static <F extends HasParts, K, R> R fold1(List<F> files, Function<F, K> keyF, BiFunction<K, Integer, R> create) {
        if (files.isEmpty()) {
            throw new IllegalArgumentException();
        }
        K first = keyF.apply(files.iterator().next());
        for (int i = 0; i < files.size(); ++i) {
            F next = files.get(i);
            K nextKey = keyF.apply(next);
            if (!first.equals(nextKey)) {
                throw new IllegalArgumentException("wrong group: " + files + ", keys  " + first + " != " + nextKey);
            }
            if (next.getPartNo() != i) {
                throw new IllegalArgumentException("wrong group: " + files + ", expected file with index " + i + " but was " + next.getPartNo());
            }
            if (next.isLast() != (i == files.size() - 1)) {
                throw new IllegalArgumentException("wrong group: " + files + ", miss file with index " + (i + 1));
            }
        }
        return create.apply(first, files.size());
    }
}
